# Before 'make install' is performed this script should be runnable with
# 'make test'. After 'make install' it should work as 'perl Lingua-RU-Numeral.t'

#########################

# change 'tests => 1' to 'tests => last_test_to_print';

use 5.010;
use strict;
use warnings;

use utf8;
use open ':std', ':encoding(utf8)';

# use Test::More 'no_plan';
use Test::More tests => 1879;
use Test::More::UTF8;

BEGIN { use_ok('Lingua::RU::Numeral') };
use Lingua::RU::Numeral qw(:all);


##### Test num2cardinal() #####

my $dt = &datest_base;

my $prolog = [];
my $epilog = {};

# Именительный, Родительный, Дательный, Винительный, Творительный, Предложный
# (n)ominative (g)enitive   (d)ative   (a)ccusative (i)nstrumental (p)repositional
for my $case ( qw{ n g d a i p } ) {
	next unless exists $dt->{ $case };

	                   # Мужской,  Женский, Средний
	for my $gender ( qw{ masculine feminine neuter } ) {
		next unless exists $dt->{ $case }{ $gender };

		my $i = 0;
		for my $number ( sort keys %{ $dt->{ $case }{ $gender } } ) {

			my $multi = $i%2 ? 'singular' : 0;

			if( $number =~/^animate/i ) {
				my $object = 'animate';

				for my $subnum ( sort { $a <=> $b } keys %{ $dt->{ $case }{ $gender }{ $object } } ) {

					# Test 'animate' tag: одушевлённый
					is( num2cardinal( $subnum,
								'case'   => $case,
								'gender' => $gender,
								'multi'  => $multi,
								'object' => $object,
								'prolog' => $prolog,
								'epilog' => $epilog,
							),
							$dt->{ $case }{ $gender }{ $object }{ $subnum },
							"number=$subnum: multi=$multi, case=$case, gender=$gender, object=$object"
						);

					++$i;
				}
				next;
			}

			my $object = 'inanimate';

			# Test 'inanimate' tag: неодушевлённый
			is( num2cardinal( $number,
							'case'   => $case,
							'gender' => $gender,
							'multi'  => $multi,
							'object' => $object,
							'prolog' => $prolog,
							'epilog' => $epilog,
						),
						$dt->{ $case }{ $gender }{ $number },
						"number=$number: multi=$multi, case=$case, gender=$gender, object=$object"
					);

			++$i;
		}
	}

	# Plural: множественное число
	if( exists $dt->{ $case }{'plural'} ) {
		my $i = 0;
		for my $number ( sort { $a <=> $b } keys %{ $dt->{ $case }{'plural'} } ) {

			            # неодушевлённый, одушевлённый
			for my $object ('inanimate', 'animate') {
				next unless exists $dt->{ $case }{'plural'}{ $number }{ $object };

				my $multi = $i%2 ? 'plural' : 1;
				my $gender = $i%3 ? $i%2 ? 'masculine' : 'feminine' : 'neuter';

				# Test 'plural' key
				is( num2cardinal( $number,
							'case'   => $case,
							'gender' => $gender,
							'multi'  => $multi,
							'object' => $object,
							'prolog' => $prolog,
							'epilog' => $epilog,
						),
						$dt->{ $case }{'plural'}{ $number }{ $object },
						"number=$number: multi=$multi, case=$case, gender=$gender, object=$object"
					);

				++$i;
			}

		}
	}

}

# Test 'prolog' key
$dt = &datest_prolog;

# Именительный, Родительный, Дательный, Винительный, Творительный, Предложный
# (n)ominative (g)enitive   (d)ative   (a)ccusative (i)nstrumental (p)repositional
for my $case ( qw{ n g d a i p } ) {
	next unless exists $dt->{ $case };

	my $prolog = $dt->{ $case }{'prolog'};

	                   # Мужской,  Женский, Средний
	for my $gender ( qw{ masculine feminine neuter } ) {
		next unless exists $dt->{ $case }{ $gender };

		my $i = 0;
		for my $number ( sort keys %{ $dt->{ $case }{ $gender } } ) {

			my $multi = $i%2 ? 'singular' : 0;

			is( num2cardinal( $number,
							'case'   => $case,
							'gender' => $gender,
							'multi'  => $multi,
							'prolog' => $prolog,
						),
						$dt->{ $case }{ $gender }{ $number },
						"number=$number: multi=$multi, case=$case, gender=$gender, prolog=". (ref($prolog) =~/ARRAY/ ? join(',', @$prolog) : $prolog )
					);

			++$i;
		}
	}

	# множественное число
	if( exists $dt->{ $case }{'plural'} ) {
		my $i = 0;
		for my $number ( sort { $a <=> $b } keys %{ $dt->{ $case }{'plural'} } ) {

			my $multi = $i%2 ? 'plural' : 1;
			my $gender = $i%3 ? $i%2 ? 'masculine' : 'feminine' : 'neuter';

			# Test 'plural'
			is( num2cardinal( $number,
							'case'   => $case,
							'gender' => $gender,
							'multi'  => $multi,
							'prolog' => $prolog,
						),
						$dt->{ $case }{'plural'}{ $number },
						"number=$number: multi=$multi, case=$case, gender=$gender, prolog=". (ref($prolog) =~/ARRAY/ ? join(',', @$prolog) : $prolog )
				);

			++$i;
		}
	}
}


# Test 'epilog' with keywords from internal reference list
$dt = &datest_epilog;

for my $epilog ( sort keys %$dt ) {

	for my $case ( qw{ n g d a i p } ) {
		next unless exists $dt->{ $epilog }{ $case };

		for my $number ( sort keys %{ $dt->{ $epilog }{ $case } } ) {

			if( $number eq 'plural') {	# Check 'plural'
				my $multi = 'plural';

				for my $subnum ( sort { $a <=> $b } keys %{ $dt->{ $epilog }{ $case }{'plural'} } ) {

					is( num2cardinal( $subnum,
							'case'   => $case,
							'multi'  => $multi,
							'epilog' => $epilog,
						),
						$dt->{ $epilog }{ $case }{'plural'}{ $subnum },
						"number=$subnum: multi=$multi, case=$case, epilog=$epilog"
					);

				}

			}
			else {	# Check 'singular' key
				my $multi = 'singular';

				is( num2cardinal( $number,
							'case'   => $case,
							'multi'  => $multi,
							'epilog' => $epilog,
						),
						$dt->{ $epilog }{ $case }{ $number },
						"number=$number: multi=$multi, case=$case, epilog=$epilog"
					);

			}

		}
	}

}

# Test 'epilog' with manual setting
my $multi = 'singular';

my $case = 'nominative';	# Именительный: кто? что?

$epilog = {
	'object' => 'inanimate',
	'gender' => 'feminine',
	'root' => ['печат'],
	'ends' => [
					['ей','ь', ('и') x 3, ('ей') x 2 ],
				],
};

my $number = 1;
is( num2cardinal( $number,
					'case'   => $case,
					'multi'  => $multi,
					'epilog' => $epilog,
				),
				'одна печать',
				"number=$number: multi=$multi, case=$case, epilog=". $epilog->{'root'}[0] . $epilog->{'ends'}[0][ $number ]
	);

$case = 'genitive';	# Родительный: кого? чего?
$epilog->{'ends'} = [ ['ей','и', ('ей') x 5 ] ];

is( num2cardinal( $number,
					'case'   => $case,
					'epilog' => $epilog,
				),
				'одной печати',
				"number=$number: multi=$multi, case=$case, epilog=". $epilog->{'root'}[0] . $epilog->{'ends'}[0][ $number ]
	);

$epilog->{'ends'} = [ [('ей') x 7 ] ];

is( num2cardinal( $number,
					'case'   => $case,
					'multi'  => 'plural',
					'epilog' => $epilog,
				),
				'одних печатей',
				"number=$number: multi=$multi, case=$case, epilog=". $epilog->{'root'}[0] . $epilog->{'ends'}[0][ $number ]
	);

$case = 'dative';	# Дательный: кому? чему?
$epilog->{'ends'} = [ ['ям','и', ('ям') x 5 ] ];

$number = 2;
is( num2cardinal( $number,
					'case'   => $case,
					'epilog' => $epilog,
				),
				'двум печатям',
				"number=$number: multi=$multi, case=$case, epilog=". $epilog->{'root'}[0] . $epilog->{'ends'}[0][ $number ]
	);

$case = 'accusative';	# Винительный: кого? что?
$epilog->{'ends'} = [ ['ей','ь', ('и') x 3, ('ей') x 2 ] ];

$number = 3;
is( num2cardinal( $number,
					'case'   => $case,
					'epilog' => $epilog,
				),
				'три печати',
				"number=$number: multi=$multi, case=$case, epilog=". $epilog->{'root'}[0] . $epilog->{'ends'}[0][ $number ]
	);

$case = 'instrumental';	# Творительный: кем? чем?
$prolog = 'c';
$epilog->{'ends'} = [ ['ями','ью', ('ями') x 5 ] ];

$number = 100;
is( num2cardinal( $number,
					'case'   => $case,
					'prolog' => $prolog,
					'epilog' => $epilog,
	),
	'со ста печатями',
	"number=$number: multi=$multi, case=$case, epilog=". $epilog->{'root'}[0] . $epilog->{'ends'}[0][-1]
);

$number = 11;
$epilog->{'ends'} = [ ['ях','и', ('ях') x 5 ] ];

# Check auto-case setting for prolog
is( num2cardinal( $number,
					'prolog' => 'о',
					'epilog' => $epilog,
	),
	'об одиннадцати печатях',
	"number=$number: multi=$multi, epilog=". $epilog->{'root'}[0] . $epilog->{'ends'}[0][-2]
);

# Custom setting for prolog
$case = 'prepositional';	# Предложный: о ком? о чём?

for my $prolog ( ['о','об=од'], {'о'=>undef, 'об'=>'од'} ) {
	is( num2cardinal( $number,
					'case'   => $case,
					'prolog' => $prolog,
					'epilog' => $epilog,
		),
		'об одиннадцати печатях',
		"number=$number: multi=$multi, case=$case, epilog=". $epilog->{'root'}[0] . $epilog->{'ends'}[0][-2]
	);

}

my @answrs = (
	'для одиннадцати',	# genitive - Родительный: кого? чего?
	'соответственно одиннадцати',	# dative - Дательный: кому? чему?
	'исключая одиннадцать',	# accusative - Винительный: кого? что?
	'по-над одиннадцатью',	# instrumental - Творительный: кем? чем?
	'при одиннадцати',	# prepositional - Предложный: ком? чём?
	'согласно одиннадцати',	# dative (Дательный, by default)
);

for my $case ( qw{ n g d a i p } ) {
	my $i = 0;
	for my $prolog ('для','соответственно','исключая','по-над','при','согласно') {
		is( num2cardinal( $number, 'case' => $case, 'prolog' => $prolog ),
			$answrs[ $i++ ],
			"number=$number: case=$case, prolog='$prolog'"
		);
	}
}

@answrs = (
	'промежду одиннадцати',	# genitive (Родительный, by default)
	'благодаря одиннадцати',	# dative (Дательный, by default)
	'за одиннадцатью',	# instrumental (Творительный, by default)
	'в одиннадцать',	# accusative - (Винительный, by default): кого? что?
	'на одиннадцать',	# accusative - (Винительный, by default): кого? что?
	'по одиннадцати',	# dative (Дательный, by default)
);
for my $case ( qw{ n g d } ) {
	my $i = 0;
	for my $prolog ('промежду','благодаря','за','в','на','по') {
		is( num2cardinal( $number, 'case' => $case, 'prolog' => $prolog ),
			$answrs[ $i++ ],
			"number=$number: case=$case, prolog='$prolog'"
		);
	}
}

@answrs = (
	'с одиннадцатью',
	'с одиннадцати',
	'с одиннадцатью',
);
my $i = 0;
for my $case ( qw{ n g d } ) {
	is( num2cardinal( $number, 'case' => $case, 'prolog' => 'со'),
		$answrs[ $i++ ],
		"number=$number: case=$case, prolog='$prolog'"
	);
}

@answrs = ( # accusative - (Винительный, by default): кого? что?
	'за одиннадцать',
	'в одиннадцать',
	'об одиннадцать',
	'на одиннадцать',
);
$i = 0;
for my $prolog ('за','в','об','на') {
	is( num2cardinal( $number, 'case' => 'a', 'prolog' => $prolog ),
		$answrs[ $i++ ],
		"number=$number: case=$case, prolog='$prolog'"
	);
}

# 'epilog' which is configured manually. Nominative case (Именительный падеж)
$epilog = {
		'object' => 'inanimate',
		'gender' => 'feminine',
		'root'   => ['норвежск','крон'],
		'ends'   => [ ['их','ая', ('их') x 5 ], ['','а', ('ы') x 3, ('') x 2] ],
};

@answrs = (
	'ноль норвежских крон',
	'одна норвежская крона',
	'две норвежских кроны',
	'три норвежских кроны',
	'четыре норвежских кроны',
	'пять норвежских крон',
	'десять норвежских крон',
	'сто норвежских крон',
	'одна тысяча норвежских крон'
);

$i = 0;
is( num2cardinal( $_, epilog => $epilog ),
	$answrs[ $i++ ],
	"number=$_: epilog='". $epilog->{'root'}[0] ."'+'". $epilog->{'root'}[1] ."'"
) for 0..5, 10, 100, 1000;


# Check 'alternatives'
is( num2cardinal( 0,
					alt => { 0=>'TRUE'},
	),
	'нуль',
	"Check zero in alternate form 'alt' => 0"
	);

is( num2cardinal( 2000,
					alt => { 0=>'TRUE'},
	),
	'две тысячи',
	"Check 2000 with 'alt' => 0"
	);

is( num2cardinal( 1000,
					case => 'i',
					alt => {'i.T'=>'Colloquial'},
	),
	'одной тысячью',
	"Check 1000 in colloquial form 'i.T'=>'C'"
	);

is( num2cardinal( 1000,
					case => 'i',
					alt => {'i.T'=>'Obsolete'},
	),
	'одной тысячею',
	"Check 1000 in obsolete form 'i.T'=>'O'"
	);

is( num2cardinal( 1000,
					case => 'i',
					alt => {'i.T'=>'TRUE'},
	),
	'одной тысячей',
	"Check 1000 in FALSE form 'i.T'=>'TRUE'"
	);


is( num2cardinal( 8000,
					case => 'i',
					alt => {'i.8'=>'TRUE'},
	),
	'восьмью тысячами',
	"Check 8000 in colloquial form 'i.8'=>'TRUE'"
	);

is( num2cardinal( 80,
					case => 'i',
					alt => {'i.8'=>'TRUE'},
	),
	'восьмьюдесятью',
	"Check 80 in colloquial form 'i.8'=>'TRUE'"
	);

is( num2cardinal( 800,
					case => 'i',
					alt => {'i.8'=>'TRUE'},
	),
	'восьмьюстами',
	"Check 800 in colloquial form 'i.8'=>'TRUE'"
	);

is( num2cardinal( 1000,
					case => 'i',
					gender => 'feminine',
					alt => {'if1'=>'TRUE'},
	),
	'одною тысячей',
	"Check 1000 in alternate form 'if1'=>'TRUE'"
	);

is( num2cardinal( 1000,
					case => 'i',
					gender => 'feminine',
					alt => {'if1'=>'TRUE', 'i.T'=>'O'},
	),
	'одною тысячею',
	"Check 1000 in alternate and obsolete form 'if1'=>'TRUE', 'i.T'=>'O'"
	);


# Check 'ucfirst'
$dt = &datest_ucfirst;

# Именительный, Родительный, Дательный, Винительный, Творительный, Предложный
# (n)ominative (g)enitive   (d)ative   (a)ccusative (i)nstrumental (p)repositional
for my $case ( qw{ n g d a i p } ) {
	next unless exists $dt->{ $case };

	my $prolog = $dt->{ $case }{'prolog'} || [];
	my $epilog = $dt->{ $case }{'epilog'} || {};

	                   # Мужской,  Женский, Средний
	for my $gender ( qw{ masculine feminine neuter } ) {
		next unless exists $dt->{ $case }{ $gender };

		for my $number ( sort keys %{ $dt->{ $case }{ $gender } } ) {

			my $multi = 'singular';

			is( num2cardinal( $number,
							'case'   => $case,
							'gender' => $gender,
							'multi'  => $multi,
							'prolog' => $prolog,
							'epilog' => $epilog,
							'ucfirst' => 1,
						),
						$dt->{ $case }{ $gender }{ $number },
						"Check 'ucfirst': number=$number, multi=$multi, case=$case, gender=$gender"
					);

		}
	}

	# множественное число
	if( exists $dt->{ $case }{'plural'} ) {
		my $i = 0;
		for my $number ( sort { $a <=> $b } keys %{ $dt->{ $case }{'plural'} } ) {

			my $multi = 'plural';
			my $gender = $i%3 ? $i%2 ? 'masculine' : 'feminine' : 'neuter';

			# Test 'plural'
			is( num2cardinal( $number,
							'case'   => $case,
							'gender' => $gender,
							'multi'  => $multi,
							'prolog' => $prolog,
							'epilog' => $epilog,
							'ucfirst' => 1,
						),
						$dt->{ $case }{'plural'}{ $number },
						"Check 'ucfirst': number=$number: multi=$multi, case=$case, gender=$gender"
				);

			++$i;
		}
	}
}

# Miscellaneous

is( num2cardinal( 1,
		'case'   => 'accusative',
		'object' => 'animate',
		),
	'одного',
	"Check case=>'accusative', object=>'animate'"
	);

is( num2cardinal('0000000000003'),
	'три',
	'Check 0000000000003'
	);

is( num2cardinal( 5, 'multi' => 'plural'),
	'пять',
	'Check plural ordinary'
	);

is( num2cardinal( 5000000, 'multi' => 'plural'),
	'пять миллионов',
	'Check plural > 999999'
	);

$epilog = {
	'object' => 'animate',
	'gender' => 'masculine',
	'root'   => ['мужчин'],
	'ends'   => [ ['','а', ('ы') x 3, ('') x 2 ] ],
};
is( num2cardinal( 2, 'epilog' => $epilog ),
	'два мужчины',
	'Check EPILOG: object => animate'
	);

delete $epilog->{'gender'};
is( num2cardinal( 2, 'epilog' => $epilog ),
	'два мужчины',
	"Check EPILOG: without 'gender' (by default)"
	);

is( case_endings( 0, 'epilog'=>'RUB'),
	'0 рублей',
	"Check case_endings sub: number = 0"
	);

is( case_endings( 1, 'epilog'=>'RUB'),
	'1 рубль',
	"Check case_endings sub: number = 1"
	);

is( case_endings( 1000, 'epilog'=>'RUB'),
	'1000 рублей',
	"Check case_endings sub: number = 1000"
	);

is( case_endings( 1, 'prolog'=>'с', 'epilog'=>'RUB'),
	'с 1 рублём',
	"Check case_endings sub: with number = 1"
	);

is( case_endings( 100, 'prolog'=>'с', 'epilog'=>'BYR'),
	'со 100 белорусскими рублями',
	"Check case_endings sub: with number = 100 BYR"
	);

is( case_endings( 1000, 'prolog'=>'за', 'epilog'=>'CNY'),
	'за 1000 юаней',
	"Check case_endings sub: number = 1000"
	);

is( spelled_out_number( 100_000, case=>'accusative', 'prolog'=>'за', 'epilog'=>'CNY', 'ucfirst'=>1 ),
#	'за 1000 (Одну тысячу) юаней',
	'за 100000 (Сто тысяч) юаней',
	"Check spelled_out_number sub: number = 100_000"
	);


# ERROR/UNDEF testing

num2cardinal( 777 );
is( $!+0,
	22,
	'num2cardinal: check for VOID context'
	);

is( num2cardinal(),
	undef,
	'num2cardinal: checking for an empty argument -- undefined as expected'
	);

case_endings( 777 );
is( $!+0,
	22,
	'case_endings: check for VOID context'
	);

is( case_endings(),
	undef,
	'case_endings: checking for an empty argument -- undefined as expected'
	);

spelled_out_number( 777 );
is( $!+0,
	22,
	'spelled_out_number: check for VOID context'
	);

is( spelled_out_number(),
	undef,
	'spelled_out_number: checking for an empty argument -- undefined as expected'
	);


$epilog = {
	'object' => 'animate',
	'gender' => 'masculine',
	'root'   => ['мужчин'],
	'ends'   => [ [('') x 7 ] ],
	'plural' => sub{ },
};
is( num2cardinal( 1,
		'case'   => 'accusative',
		'multi' => 'plural',
		'epilog' => $epilog,
		),
	'одних мужчин',
	"Check wrong EPILOG: specified sub() for 'plural' key"
	);

is( num2cardinal('blah'),
	'ноль',
	'Check wrong NUM: blah'
	);

is( num2cardinal( 0, 'case' => 'blah'),
	'ноль',
	'Check wrong CASE'
	);

is( num2cardinal( 0, 'gender' => 'bear'),
	'ноль',
	'Check wrong GENDER'
	);

is( num2cardinal( 0, 'object' => 'rubbish'),
	'ноль',
	'Check wrong OBJECT'
	);

is( num2cardinal( 0, 'multi' => 'blah'),
	'ноль',
	'Check wrong MULTI'
	);

is( num2cardinal( 1, 'prolog' => {''=>undef} ),
	'один',
	'Check wrong PROLOG: empty hash key is undef'
	);

is( num2cardinal( 1, 'prolog' => {''=>'в'} ),
	'один',
	'Check wrong PROLOG: empty hash key is defined'
	);

is( num2cardinal( 0, 'prolog' => [undef,''] ),
	'ноль',
	'Check wrong PROLOG: undef for 0'
	);

is( num2cardinal( 10, 'prolog' => [''] ),
	'десять',
	'Check wrong PROLOG: undef for 10'
	);

is( num2cardinal( 7, 'prolog' => [undef] ),
	'семь',
	'Check wrong PROLOG: undef for 7'
	);

is( num2cardinal( 0, 'prolog' => ['в','на'] ),
	'в ноль',
	'Check PROLOG: wrong array[1]'
	);

is( num2cardinal( 0, 'prolog' => sub{ } ),
	'ноль',
	'Check wrong PROLOG: specified sub'
	);

is( num2cardinal( 0, 'prolog' => 'blah'),
	'ноль',
	'Check wrong PROLOG: unknown tag'
	);


$epilog = {
	'gender' => 'masculine',
	'root' => ['час'],
	'ends' => [
					['ов','', ('а') x 3, ('ов') x 2 ],
				],
};
is( num2cardinal( 0, 'epilog' => $epilog ),
	'ноль часов',
	'Check EPILOG: empty object'
	);

$epilog->{'gender'} = 'rubbish';
is( num2cardinal( 0, 'epilog' => $epilog ),
	'ноль часов',
	'Check EPILOG: wrong gender'
	);

is( num2cardinal( 0, 'epilog' => sub{ } ),
	'ноль',
	'Check wrong EPILOG: specified sub'
	);


exit;


sub datest_base {
	return {
		'n' => { # nominative (Именительный падеж): кто? что?
			'masculine' => { # Мужской род
				'0'  => 'ноль',
				'1'  => 'один',
				'2'  => 'два',
				'3'  => 'три',
				'4'  => 'четыре',
				'5'  => 'пять',
				'6'  => 'шесть',
				'7'  => 'семь',
				'8'  => 'восемь',
				'9'  => 'девять',
				'10' => 'десять',
				'11' => 'одиннадцать',
				'12' => 'двенадцать',
				'13' => 'тринадцать',
				'14' => 'четырнадцать',
				'15' => 'пятнадцать',
				'16' => 'шестнадцать',
				'17' => 'семнадцать',
				'18' => 'восемнадцать',
				'19' => 'девятнадцать',

				'20' => 'двадцать',
				'30' => 'тридцать',
				'40' => 'сорок',
				'50' => 'пятьдесят',
				'60' => 'шестьдесят',
				'70' => 'семьдесят',
				'80' => 'восемьдесят',
				'90' => 'девяносто',

				'100' => 'сто',
				'200' => 'двести',
				'300' => 'триста',
				'400' => 'четыреста',
				'500' => 'пятьсот',
				'600' => 'шестьсот',
				'700' => 'семьсот',
				'800' => 'восемьсот',
				'900' => 'девятьсот',

				'1000' => 'одна тысяча',
				'2000' => 'две тысячи',
				'3000' => 'три тысячи',
				'4000' => 'четыре тысячи',
				'5000' => 'пять тысяч',
				'10000' => 'десять тысяч',
				'90000' => 'девяносто тысяч',
				'100000' => 'сто тысяч',
				'900000' => 'девятьсот тысяч',

				'1000000' => 'один миллион',
				'2000000' => 'два миллиона',
				'3000000' => 'три миллиона',
				'4000000' => 'четыре миллиона',
				'5000000' => 'пять миллионов',
				'10000000' => 'десять миллионов',
				'90000000' => 'девяносто миллионов',
				'100000000' => 'сто миллионов',
				'900000000' => 'девятьсот миллионов',

				'1000000000' => 'один миллиард',
				'2000000000' => 'два миллиарда',
				'3000000000' => 'три миллиарда',
				'4000000000' => 'четыре миллиарда',
				'5000000000' => 'пять миллиардов',
				'10000000000' => 'десять миллиардов',
				'90000000000' => 'девяносто миллиардов',
				'100000000000' => 'сто миллиардов',
				'900000000000' => 'девятьсот миллиардов',

				'1000000000000' => 'один триллион',
				'2000000000000' => 'два триллиона',
				'3000000000000' => 'три триллиона',
				'4000000000000' => 'четыре триллиона',
				'5000000000000' => 'пять триллионов',
				'10000000000000' => 'десять триллионов',
				'90000000000000' => 'девяносто триллионов',
				'100000000000000' => 'сто триллионов',
				'900000000000000' => 'девятьсот триллионов',

				'1002' => 'одна тысяча два',
				'1000003' => 'один миллион три',
				'1000000004' => 'один миллиард четыре',
				'1000000000005' => 'один триллион пять',
				'2006000' => 'два миллиона шесть тысяч',
				'2000007000' => 'два миллиарда семь тысяч',
				'2000000002000' => 'два триллиона две тысячи',
				'5008000000' => 'пять миллиардов восемь миллионов',
				'6000030000000' => 'шесть триллионов тридцать миллионов',
				'1007000000000' => 'один триллион семь миллиардов',

				'930651342187216' => 'девятьсот тридцать триллионов шестьсот пятьдесят один миллиард триста сорок два миллиона сто восемьдесят семь тысяч двести шестнадцать',
				'999999999999999' => 'девятьсот девяносто девять триллионов девятьсот девяносто девять миллиардов девятьсот девяносто девять миллионов девятьсот девяносто девять тысяч девятьсот девяносто девять',
				'1000000000000000' => '1000000000000000 > 999_999_999_999_999 !',
			},
			'feminine' => { # Женский род
				'0'  => 'ноль',
				'1'  => 'одна',
				'2'  => 'две',
				'3'  => 'три',	# like 'masculine'
				'4'  => 'четыре',	# ...
				'5'  => 'пять',	# ...
				'10' => 'десять',	# ...

				'100' => 'сто',
				'900' => 'девятьсот',

				'1000' => 'одна тысяча',
				'2000' => 'две тысячи',
				'90000' => 'девяносто тысяч',
				'100000' => 'сто тысяч',

				'1000000' => 'один миллион',
				'2000000' => 'два миллиона',
				'90000000' => 'девяносто миллионов',
				'100000000' => 'сто миллионов',

				'1000000000' => 'один миллиард',
				'2000000000' => 'два миллиарда',
				'90000000000' => 'девяносто миллиардов',
				'100000000000' => 'сто миллиардов',

				'1000000000000' => 'один триллион',
				'2000000000000' => 'два триллиона',
				'90000000000000' => 'девяносто триллионов',
				'100000000000000' => 'сто триллионов',

				'101101102101101' => 'сто один триллион сто один миллиард сто два миллиона сто одна тысяча сто одна',

			},
			'neuter' => { # Средний род
				'0'  => 'ноль',
				'1'  => 'одно',
				'2'  => 'два',	# like 'masculine'
				'3'  => 'три',	# ...
				'4'  => 'четыре',	# ...
				'5'  => 'пять',	# ...
				'10' => 'десять',	# ...

				'100' => 'сто',
				'900' => 'девятьсот',

				'1000' => 'одна тысяча',
				'2000' => 'две тысячи',
				'90000' => 'девяносто тысяч',
				'100000' => 'сто тысяч',

				'1000000' => 'один миллион',
				'2000000' => 'два миллиона',
				'90000000' => 'девяносто миллионов',
				'100000000' => 'сто миллионов',

				'1000000000' => 'один миллиард',
				'2000000000' => 'два миллиарда',
				'90000000000' => 'девяносто миллиардов',
				'100000000000' => 'сто миллиардов',

				'1000000000000' => 'один триллион',
				'2000000000000' => 'два триллиона',
				'90000000000000' => 'девяносто триллионов',
				'100000000000000' => 'сто триллионов',

				'101101102101101' => 'сто один триллион сто один миллиард сто два миллиона сто одна тысяча сто одно',
			},
			'plural' => {	# множественное число
				'1' => {'inanimate' => 'одни'}, # неодушевлённый, animate(одушевлённый)
				'1000' => {'inanimate' => 'одни тысячи'},
				'1000000' => {'inanimate' => 'одни миллионы'},
				'1000000000' => {'inanimate' => 'одни миллиарды'},
				'1000000000000' => {'inanimate' => 'одни триллионы'},
			},

		}, #--------------------------------------------
		'g' => { # genitive (Родительный падеж): кого? чего?
			'masculine' => { # Мужской род
				'0'  => 'ноля',
				'1'  => 'одного',
				'2'  => 'двух',
				'3'  => 'трёх',
				'4'  => 'четырёх',
				'5'  => 'пяти',
				'6'  => 'шести',
				'7'  => 'семи',
				'8'  => 'восьми',
				'9'  => 'девяти',
				'10' => 'десяти',
				'11' => 'одиннадцати',
				'12' => 'двенадцати',
				'13' => 'тринадцати',
				'14' => 'четырнадцати',
				'15' => 'пятнадцати',
				'16' => 'шестнадцати',
				'17' => 'семнадцати',
				'18' => 'восемнадцати',
				'19' => 'девятнадцати',

				'20' => 'двадцати',
				'30' => 'тридцати',
				'40' => 'сорока',
				'50' => 'пятидесяти',
				'60' => 'шестидесяти',
				'70' => 'семидесяти',
				'80' => 'восьмидесяти',
				'90' => 'девяноста',

				'100' => 'ста',
				'200' => 'двухсот',
				'300' => 'трёхсот',
				'400' => 'четырёхсот',
				'500' => 'пятисот',
				'600' => 'шестисот',
				'700' => 'семисот',
				'800' => 'восьмисот',
				'900' => 'девятисот',

				'1000' => 'одной тысячи',
				'2000' => 'двух тысяч',
				'3000' => 'трёх тысяч',
				'4000' => 'четырёх тысяч',
				'5000' => 'пяти тысяч',
				'10000' => 'десяти тысяч',
				'90000' => 'девяноста тысяч',
				'100000' => 'ста тысяч',
				'900000' => 'девятисот тысяч',

				'1000000' => 'одного миллиона',
				'2000000' => 'двух миллионов',
				'3000000' => 'трёх миллионов',
				'4000000' => 'четырёх миллионов',
				'5000000' => 'пяти миллионов',
				'10000000' => 'десяти миллионов',
				'90000000' => 'девяноста миллионов',
				'100000000' => 'ста миллионов',
				'900000000' => 'девятисот миллионов',

				'1000000000' => 'одного миллиарда',
				'2000000000' => 'двух миллиардов',
				'3000000000' => 'трёх миллиардов',
				'4000000000' => 'четырёх миллиардов',
				'5000000000' => 'пяти миллиардов',
				'10000000000' => 'десяти миллиардов',
				'90000000000' => 'девяноста миллиардов',
				'100000000000' => 'ста миллиардов',
				'900000000000' => 'девятисот миллиардов',

				'1000000000000' => 'одного триллиона',
				'2000000000000' => 'двух триллионов',
				'3000000000000' => 'трёх триллионов',
				'4000000000000' => 'четырёх триллионов',
				'5000000000000' => 'пяти триллионов',
				'10000000000000' => 'десяти триллионов',
				'90000000000000' => 'девяноста триллионов',
				'100000000000000' => 'ста триллионов',
				'900000000000000' => 'девятисот триллионов',

				'1002' => 'одной тысячи двух',
				'1000003' => 'одного миллиона трёх',
				'1000000004' => 'одного миллиарда четырёх',
				'1000000000005' => 'одного триллиона пяти',
				'2006000' => 'двух миллионов шести тысяч',
				'2000007000' => 'двух миллиардов семи тысяч',
				'2000000002000' => 'двух триллионов двух тысяч',
				'5008000000' => 'пяти миллиардов восьми миллионов',
				'6000030000000' => 'шести триллионов тридцати миллионов',
				'1007000000000' => 'одного триллиона семи миллиардов',

				'930651342187216' => 'девятисот тридцати триллионов шестисот пятидесяти одного миллиарда трёхсот сорока двух миллионов ста восьмидесяти семи тысяч двухсот шестнадцати',
				'999999999999999' => 'девятисот девяноста девяти триллионов девятисот девяноста девяти миллиардов девятисот девяноста девяти миллионов девятисот девяноста девяти тысяч девятисот девяноста девяти',

			},
			'feminine' => { # Женский род
				'0'  => 'ноля',	# ...
				'1'  => 'одной',
				'2'  => 'двух',	# like 'masculine'
				'3'  => 'трёх',	# ...
				'4'  => 'четырёх',	# ...
				'5'  => 'пяти',	# ...
				'10' => 'десяти',	# ...

				'100' => 'ста',
				'900' => 'девятисот',

				'1000' => 'одной тысячи',
				'2000' => 'двух тысяч',
				'90000' => 'девяноста тысяч',
				'100000' => 'ста тысяч',

				'1000000' => 'одного миллиона',
				'2000000' => 'двух миллионов',
				'90000000' => 'девяноста миллионов',
				'100000000' => 'ста миллионов',

				'1000000000' => 'одного миллиарда',
				'2000000000' => 'двух миллиардов',
				'90000000000' => 'девяноста миллиардов',
				'100000000000' => 'ста миллиардов',

				'1000000000000' => 'одного триллиона',
				'2000000000000' => 'двух триллионов',
				'90000000000000' => 'девяноста триллионов',
				'100000000000000' => 'ста триллионов',

				'101101102101101' => 'ста одного триллиона ста одного миллиарда ста двух миллионов ста одной тысячи ста одной',

			},
			'neuter' => { # Средний род
				'0'  => 'ноля',	# like 'masculine'
				'1'  => 'одного',	# ...
				'2'  => 'двух',	# ...
				'3'  => 'трёх',	# ...
				'4'  => 'четырёх',	# ...
				'5'  => 'пяти',	# ...
				'10' => 'десяти',	# ...

				'100' => 'ста',
				'900' => 'девятисот',

				'1000' => 'одной тысячи',
				'2000' => 'двух тысяч',
				'90000' => 'девяноста тысяч',
				'100000' => 'ста тысяч',

				'1000000' => 'одного миллиона',
				'2000000' => 'двух миллионов',
				'90000000' => 'девяноста миллионов',
				'100000000' => 'ста миллионов',

				'1000000000' => 'одного миллиарда',
				'2000000000' => 'двух миллиардов',
				'90000000000' => 'девяноста миллиардов',
				'100000000000' => 'ста миллиардов',

				'1000000000000' => 'одного триллиона',
				'2000000000000' => 'двух триллионов',
				'90000000000000' => 'девяноста триллионов',
				'100000000000000' => 'ста триллионов',

				'101101102101101' => 'ста одного триллиона ста одного миллиарда ста двух миллионов ста одной тысячи ста одного',

			},

			'plural' => {	# множественное число
				'1' => {'inanimate' => 'одних'}, # неодушевлённый, animate(одушевлённый)
				'1000' => {'inanimate' => 'одних тысяч'},
				'1000000' => {'inanimate' => 'одних миллионов'},
				'1000000000' => {'inanimate' => 'одних миллиардов'},
				'1000000000000' => {'inanimate' => 'одних триллионов'},
			},

		}, #--------------------------------------------
		'd' => { # dative (Дательный падеж): кому? чему?
			'masculine' => { # Мужской род, default
				'0'  => 'нолю',
				'1'  => 'одному',
				'2'  => 'двум',
				'3'  => 'трём',
				'4'  => 'четырём',
				'5'  => 'пяти',
				'6'  => 'шести',
				'7'  => 'семи',
				'8'  => 'восьми',
				'9'  => 'девяти',
				'10' => 'десяти',
				'11' => 'одиннадцати',
				'12' => 'двенадцати',
				'13' => 'тринадцати',
				'14' => 'четырнадцати',
				'15' => 'пятнадцати',
				'16' => 'шестнадцати',
				'17' => 'семнадцати',
				'18' => 'восемнадцати',
				'19' => 'девятнадцати',

				'20' => 'двадцати',
				'30' => 'тридцати',
				'40' => 'сорока',
				'50' => 'пятидесяти',
				'60' => 'шестидесяти',
				'70' => 'семидесяти',
				'80' => 'восьмидесяти',
				'90' => 'девяноста',

				'100' => 'ста',
				'200' => 'двумстам',
				'300' => 'трёхстам',
				'400' => 'четырёхстам',
				'500' => 'пятистам',
				'600' => 'шестистам',
				'700' => 'семистам',
				'800' => 'восьмистам',
				'900' => 'девятистам',

				'1000' => 'одной тысяче',
				'2000' => 'двум тысячам',
				'3000' => 'трём тысячам',
				'4000' => 'четырём тысячам',
				'5000' => 'пяти тысячам',
				'10000' => 'десяти тысячам',
				'90000' => 'девяноста тысячам',
				'100000' => 'ста тысячам',
				'900000' => 'девятистам тысячам',

				'1000000' => 'одному миллиону',
				'2000000' => 'двум миллионам',
				'3000000' => 'трём миллионам',
				'4000000' => 'четырём миллионам',
				'5000000' => 'пяти миллионам',
				'10000000' => 'десяти миллионам',
				'90000000' => 'девяноста миллионам',
				'100000000' => 'ста миллионам',
				'900000000' => 'девятистам миллионам',

				'1000000000' => 'одному миллиарду',
				'2000000000' => 'двум миллиардам',
				'3000000000' => 'трём миллиардам',
				'4000000000' => 'четырём миллиардам',
				'5000000000' => 'пяти миллиардам',
				'10000000000' => 'десяти миллиардам',
				'90000000000' => 'девяноста миллиардам',
				'100000000000' => 'ста миллиардам',
				'900000000000' => 'девятистам миллиардам',

				'1000000000000' => 'одному триллиону',
				'2000000000000' => 'двум триллионам',
				'3000000000000' => 'трём триллионам',
				'4000000000000' => 'четырём триллионам',
				'5000000000000' => 'пяти триллионам',
				'10000000000000' => 'десяти триллионам',
				'90000000000000' => 'девяноста триллионам',
				'100000000000000' => 'ста триллионам',
				'900000000000000' => 'девятистам триллионам',

				'1002' => 'одной тысяче двум',
				'1000003' => 'одному миллиону трём',
				'1000000004' => 'одному миллиарду четырём',
				'1000000000005' => 'одному триллиону пяти',
				'2006000' => 'двум миллионам шести тысячам',
				'2000007000' => 'двум миллиардам семи тысячам',
				'2000000002000' => 'двум триллионам двум тысячам',
				'5008000000' => 'пяти миллиардам восьми миллионам',
				'6000030000000' => 'шести триллионам тридцати миллионам',
				'1007000000000' => 'одному триллиону семи миллиардам',

				'930651342187216' => 'девятистам тридцати триллионам шестистам пятидесяти одному миллиарду трёхстам сорока двум миллионам ста восьмидесяти семи тысячам двумстам шестнадцати',
				'999999999999999' => 'девятистам девяноста девяти триллионам девятистам девяноста девяти миллиардам девятистам девяноста девяти миллионам девятистам девяноста девяти тысячам девятистам девяноста девяти',

			},
			'feminine' => { # Женский род
				'0'  => 'нолю',	# like 'masculine'
				'1'  => 'одной',
				'2'  => 'двум',	# like 'masculine'
				'3'  => 'трём',	# ...
				'4'  => 'четырём',	# ...
				'5'  => 'пяти',	# ...
				'10' => 'десяти',	# ...

				'100' => 'ста',
				'900' => 'девятистам',

				'1000' => 'одной тысяче',
				'2000' => 'двум тысячам',
				'90000' => 'девяноста тысячам',
				'100000' => 'ста тысячам',

				'1000000' => 'одному миллиону',
				'2000000' => 'двум миллионам',
				'90000000' => 'девяноста миллионам',
				'100000000' => 'ста миллионам',

				'1000000000' => 'одному миллиарду',
				'2000000000' => 'двум миллиардам',
				'90000000000' => 'девяноста миллиардам',
				'100000000000' => 'ста миллиардам',

				'1000000000000' => 'одному триллиону',
				'2000000000000' => 'двум триллионам',
				'90000000000000' => 'девяноста триллионам',
				'100000000000000' => 'ста триллионам',

				'101101102101101' => 'ста одному триллиону ста одному миллиарду ста двум миллионам ста одной тысяче ста одной',

			},
			'neuter' => { # Средний род
				'0'  => 'нолю',	# like 'masculine'
				'1'  => 'одному',	# ...
				'2'  => 'двум',	# ...
				'3'  => 'трём',	# ...
				'4'  => 'четырём',	# ...
				'5'  => 'пяти',	# ...
				'10' => 'десяти',	# ...

				'100' => 'ста',
				'900' => 'девятистам',

				'1000' => 'одной тысяче',
				'2000' => 'двум тысячам',
				'90000' => 'девяноста тысячам',
				'100000' => 'ста тысячам',

				'1000000' => 'одному миллиону',
				'2000000' => 'двум миллионам',
				'90000000' => 'девяноста миллионам',
				'100000000' => 'ста миллионам',

				'1000000000' => 'одному миллиарду',
				'2000000000' => 'двум миллиардам',
				'90000000000' => 'девяноста миллиардам',
				'100000000000' => 'ста миллиардам',

				'1000000000000' => 'одному триллиону',
				'2000000000000' => 'двум триллионам',
				'90000000000000' => 'девяноста триллионам',
				'100000000000000' => 'ста триллионам',

				'101101102101101' => 'ста одному триллиону ста одному миллиарду ста двум миллионам ста одной тысяче ста одному',
			},

			'plural' => {	# множественное число
				'1' => {'inanimate' => 'одним'}, # неодушевлённый, animate(одушевлённый)
				'1000' => {'inanimate' => 'одним тысячам'},
				'1000000' => {'inanimate' => 'одним миллионам'},
				'1000000000' => {'inanimate' => 'одним миллиардам'},
				'1000000000000' => {'inanimate' => 'одним триллионам'},
			},

		}, #--------------------------------------------
		'a' => { # accusative (Винительный падеж): inanimate (неодушевлённый объект)
			'masculine' => { # Мужской род, default : что? кого?
				'0'  => 'ноль',	# 0..19 : неодушевлённый объект, как Именительный падеж
				'1'  => 'один',
				'2'  => 'два',
				'3'  => 'три',
				'4'  => 'четыре',
				'5'  => 'пять',
				'6'  => 'шесть',
				'7'  => 'семь',
				'8'  => 'восемь',
				'9'  => 'девять',
				'10' => 'десять',
				'11' => 'одиннадцать',
				'12' => 'двенадцать',
				'13' => 'тринадцать',
				'14' => 'четырнадцать',
				'15' => 'пятнадцать',
				'16' => 'шестнадцать',
				'17' => 'семнадцать',
				'18' => 'восемнадцать',
				'19' => 'девятнадцать',

				'20' => 'двадцать',
				'30' => 'тридцать',
				'40' => 'сорок',
				'50' => 'пятьдесят',
				'60' => 'шестьдесят',
				'70' => 'семьдесят',
				'80' => 'восемьдесят',
				'90' => 'девяносто',

				'100' => 'сто',
				'200' => 'двести',
				'300' => 'триста',
				'400' => 'четыреста',
				'500' => 'пятьсот',
				'600' => 'шестьсот',
				'700' => 'семьсот',
				'800' => 'восемьсот',
				'900' => 'девятьсот',

				'1000' => 'одну тысячу',
				'2000' => 'две тысячи',
				'3000' => 'три тысячи',
				'4000' => 'четыре тысячи',
				'5000' => 'пять тысяч',
				'10000' => 'десять тысяч',
				'90000' => 'девяносто тысяч',
				'100000' => 'сто тысяч',
				'900000' => 'девятьсот тысяч',

				'1000000' => 'один миллион',
				'2000000' => 'два миллиона',
				'3000000' => 'три миллиона',
				'4000000' => 'четыре миллиона',
				'5000000' => 'пять миллионов',
				'10000000' => 'десять миллионов',
				'90000000' => 'девяносто миллионов',
				'100000000' => 'сто миллионов',
				'900000000' => 'девятьсот миллионов',

				'1000000000' => 'один миллиард',
				'2000000000' => 'два миллиарда',
				'3000000000' => 'три миллиарда',
				'4000000000' => 'четыре миллиарда',
				'5000000000' => 'пять миллиардов',
				'10000000000' => 'десять миллиардов',
				'90000000000' => 'девяносто миллиардов',
				'100000000000' => 'сто миллиардов',
				'900000000000' => 'девятьсот миллиардов',

				'1000000000000' => 'один триллион',
				'2000000000000' => 'два триллиона',
				'3000000000000' => 'три триллиона',
				'4000000000000' => 'четыре триллиона',
				'5000000000000' => 'пять триллионов',
				'10000000000000' => 'десять триллионов',
				'90000000000000' => 'девяносто триллионов',
				'100000000000000' => 'сто триллионов',
				'900000000000000' => 'девятьсот триллионов',

				'1002' => 'одну тысячу два',
				'1000003' => 'один миллион три',
				'1000000004' => 'один миллиард четыре',
				'1000000000005' => 'один триллион пять',
				'2006000' => 'два миллиона шесть тысяч',
				'2000007000' => 'два миллиарда семь тысяч',
				'2000000002000' => 'два триллиона две тысячи',
				'5008000000' => 'пять миллиардов восемь миллионов',
				'6000030000000' => 'шесть триллионов тридцать миллионов',
				'1007000000000' => 'один триллион семь миллиардов',

				'930651342187216' => 'девятьсот тридцать триллионов шестьсот пятьдесят один миллиард триста сорок два миллиона сто восемьдесят семь тысяч двести шестнадцать',
				'999999999999999' => 'девятьсот девяносто девять триллионов девятьсот девяносто девять миллиардов девятьсот девяносто девять миллионов девятьсот девяносто девять тысяч девятьсот девяносто девять',

				'animate' => {
				 # 0..19 : одушевлённый объект Мужского рода, как Родительный падеж (0..4), Именительный падеж (5..19)
					'0'  => 'ноля',
					'1'  => 'одного',
					'2'  => 'двух',
					'3'  => 'трёх',
					'4'  => 'четырёх',

					'5'  => 'пять',
					'6'  => 'шесть',
					'7'  => 'семь',
					'8'  => 'восемь',
					'9'  => 'девять',
					'10' => 'десять',
					'11' => 'одиннадцать',
					'12' => 'двенадцать',
					'13' => 'тринадцать',
					'14' => 'четырнадцать',
					'15' => 'пятнадцать',
					'16' => 'шестнадцать',
					'17' => 'семнадцать',
					'18' => 'восемнадцать',
					'19' => 'девятнадцать',

					'20' => 'двадцать',
					'30' => 'тридцать',
					'40' => 'сорок',
					'50' => 'пятьдесят',
					'60' => 'шестьдесят',
					'70' => 'семьдесят',
					'80' => 'восемьдесят',
					'90' => 'девяносто',

					'100' => 'сто',
					'200' => 'двести',
					'300' => 'триста',
					'400' => 'четыреста',
					'500' => 'пятьсот',
					'600' => 'шестьсот',
					'700' => 'семьсот',
					'800' => 'восемьсот',
					'900' => 'девятьсот',

					'1000' => 'одну тысячу',
					'2000' => 'две тысячи',
					'3000' => 'три тысячи',
					'4000' => 'четыре тысячи',
					'5000' => 'пять тысяч',
					'10000' => 'десять тысяч',
					'90000' => 'девяносто тысяч',
					'100000' => 'сто тысяч',
					'900000' => 'девятьсот тысяч',

					'1000000' => 'один миллион',
					'2000000' => 'два миллиона',
					'3000000' => 'три миллиона',
					'4000000' => 'четыре миллиона',
					'5000000' => 'пять миллионов',
					'10000000' => 'десять миллионов',
					'90000000' => 'девяносто миллионов',
					'100000000' => 'сто миллионов',
					'900000000' => 'девятьсот миллионов',

					'1000000000' => 'один миллиард',
					'2000000000' => 'два миллиарда',
					'3000000000' => 'три миллиарда',
					'4000000000' => 'четыре миллиарда',
					'5000000000' => 'пять миллиардов',
					'10000000000' => 'десять миллиардов',
					'90000000000' => 'девяносто миллиардов',
					'100000000000' => 'сто миллиардов',
					'900000000000' => 'девятьсот миллиардов',

					'1000000000000' => 'один триллион',
					'2000000000000' => 'два триллиона',
					'3000000000000' => 'три триллиона',
					'4000000000000' => 'четыре триллиона',
					'5000000000000' => 'пять триллионов',
					'10000000000000' => 'десять триллионов',
					'90000000000000' => 'девяносто триллионов',
					'100000000000000' => 'сто триллионов',
					'900000000000000' => 'девятьсот триллионов',

					'1002' => 'одну тысячу двух',
					'1000003' => 'один миллион трёх',

					'1000000004' => 'один миллиард четырёх',
					'1000000000005' => 'один триллион пять',
					'2006000' => 'два миллиона шесть тысяч',
					'2000007000' => 'два миллиарда семь тысяч',
					'2000000002000' => 'два триллиона две тысячи',
					'5008000000' => 'пять миллиардов восемь миллионов',
					'6000030000000' => 'шесть триллионов тридцать миллионов',
					'1007000000000' => 'один триллион семь миллиардов',

					'930651342187216' => 'девятьсот тридцать триллионов шестьсот пятьдесят один миллиард триста сорок два миллиона сто восемьдесят семь тысяч двести шестнадцать',
					'999999999999999' => 'девятьсот девяносто девять триллионов девятьсот девяносто девять миллиардов девятьсот девяносто девять миллионов девятьсот девяносто девять тысяч девятьсот девяносто девять',
				},

			},
			'feminine' => { # Женский род, неодушевлённый объект: 'одну','две', остальные как для Мужского рода
				'0'  => 'ноль',
				'1'  => 'одну',
				'2'  => 'две',
				'3'  => 'три',
				'4'  => 'четыре',
				'5'  => 'пять',
				'10' => 'десять',

				'100' => 'сто',
				'900' => 'девятьсот',

				'1000' => 'одну тысячу',
				'2000' => 'две тысячи',
				'90000' => 'девяносто тысяч',
				'100000' => 'сто тысяч',

				'1000000' => 'один миллион',
				'2000000' => 'два миллиона',
				'90000000' => 'девяносто миллионов',
				'100000000' => 'сто миллионов',

				'1000000000' => 'один миллиард',
				'2000000000' => 'два миллиарда',
				'90000000000' => 'девяносто миллиардов',
				'100000000000' => 'сто миллиардов',

				'1000000000000' => 'один триллион',
				'2000000000000' => 'два триллиона',
				'90000000000000' => 'девяносто триллионов',
				'100000000000000' => 'сто триллионов',

				'101101102101101' => 'сто один триллион сто один миллиард сто два миллиона сто одну тысячу сто одну',

				'animate' => { # одушевлённый объект: 'одну', остальные как для одушевлённого Мужского рода
					'0'  => 'ноля',
					'1'  => 'одну',
					'2'  => 'двух',
					'3'  => 'трёх',
					'4'  => 'четырёх',
					'5'  => 'пять',
					'10' => 'десять',

					'100' => 'сто',
					'900' => 'девятьсот',

					'1000' => 'одну тысячу',
					'2000' => 'две тысячи',
					'3000' => 'три тысячи',
					'4000' => 'четыре тысячи',
					'5000' => 'пять тысяч',
					'90000' => 'девяносто тысяч',
					'100000' => 'сто тысяч',

					'1000000' => 'один миллион',
					'2000000' => 'два миллиона',
					'3000000' => 'три миллиона',
					'4000000' => 'четыре миллиона',
					'5000000' => 'пять миллионов',
					'90000000' => 'девяносто миллионов',
					'100000000' => 'сто миллионов',

					'1000000000' => 'один миллиард',
					'2000000000' => 'два миллиарда',
					'3000000000' => 'три миллиарда',
					'4000000000' => 'четыре миллиарда',
					'5000000000' => 'пять миллиардов',
					'90000000000' => 'девяносто миллиардов',
					'100000000000' => 'сто миллиардов',

					'1000000000000' => 'один триллион',
					'2000000000000' => 'два триллиона',
					'3000000000000' => 'три триллиона',
					'4000000000000' => 'четыре триллиона',
					'5000000000000' => 'пять триллионов',
					'90000000000000' => 'девяносто триллионов',
					'100000000000000' => 'сто триллионов',

					'101101102101101' => 'сто один триллион сто один миллиард сто два миллиона сто одну тысячу сто одну',
				},

			},
			'neuter' => { # Средний род, неодушевлённый объект. 'одно', остальные как для Мужского рода
				'0'  => 'ноль',	# like 'masculine'
				'1'  => 'одно',
				'2'  => 'два',	# ...
				'3'  => 'три',	# ...
				'4'  => 'четыре',	# ...
				'5'  => 'пять',	# ...
				'10' => 'десять',	# ...

				'100' => 'сто',
				'900' => 'девятьсот',

				'1000' => 'одну тысячу',
				'2000' => 'две тысячи',
				'90000' => 'девяносто тысяч',
				'100000' => 'сто тысяч',

				'1000000' => 'один миллион',
				'2000000' => 'два миллиона',
				'90000000' => 'девяносто миллионов',
				'100000000' => 'сто миллионов',

				'1000000000' => 'один миллиард',
				'2000000000' => 'два миллиарда',
				'90000000000' => 'девяносто миллиардов',
				'100000000000' => 'сто миллиардов',

				'1000000000000' => 'один триллион',
				'2000000000000' => 'два триллиона',
				'90000000000000' => 'девяносто триллионов',
				'100000000000000' => 'сто триллионов',

				'101101102101101' => 'сто один триллион сто один миллиард сто два миллиона сто одну тысячу сто одно',

				'animate' => { # одушевлённый объект: 'одно','два', а остальные как для одушевлённого Мужского рода
					'0'  => 'ноля',
					'1'  => 'одно',	###
					'2'  => 'два',	###
					'3'  => 'трёх',
					'4'  => 'четырёх',
					'5'  => 'пять',
					'10' => 'десять',

					'100' => 'сто',
					'900' => 'девятьсот',

					'1000' => 'одну тысячу',
					'2000' => 'две тысячи',
					'3000' => 'три тысячи',
					'4000' => 'четыре тысячи',
					'5000' => 'пять тысяч',
					'90000' => 'девяносто тысяч',
					'100000' => 'сто тысяч',

					'1000000' => 'один миллион',
					'2000000' => 'два миллиона',
					'3000000' => 'три миллиона',
					'4000000' => 'четыре миллиона',
					'5000000' => 'пять миллионов',
					'90000000' => 'девяносто миллионов',
					'100000000' => 'сто миллионов',

					'1000000000' => 'один миллиард',
					'2000000000' => 'два миллиарда',
					'3000000000' => 'три миллиарда',
					'4000000000' => 'четыре миллиарда',
					'5000000000' => 'пять миллиардов',
					'90000000000' => 'девяносто миллиардов',
					'100000000000' => 'сто миллиардов',

					'1000000000000' => 'один триллион',
					'2000000000000' => 'два триллиона',
					'3000000000000' => 'три триллиона',
					'4000000000000' => 'четыре триллиона',
					'5000000000000' => 'пять триллионов',
					'90000000000000' => 'девяносто триллионов',
					'100000000000000' => 'сто триллионов',

					'101101102101101' => 'сто один триллион сто один миллиард сто два миллиона сто одну тысячу сто одно',
				},
			},

			'plural' => {	# множественное число
				'1' => {'inanimate' => 'одни', 'animate' => 'одних'}, # неодушевлённый, одушевлённый
				'1000' => {'inanimate' => 'одни тысячи', 'animate' => 'одних тысяч'},
				'1000000' => {'inanimate' => 'одни миллионы', 'animate' => 'одних миллионов'},
				'1000000000' => {'inanimate' => 'одни миллиарды', 'animate' => 'одних миллиардов'},
				'1000000000000' => {'inanimate' => 'одни триллионы', 'animate' => 'одних триллионов'},
			},

		}, #--------------------------------------------
		'i' => { # instrumental (Творительный падеж) : кем? чем?
			'masculine' => { # Мужской род, default
				'0'  => 'нолём',
				'1'  => 'одним',
				'2'  => 'двумя',
				'3'  => 'тремя',
				'4'  => 'четырьмя',
				'5'  => 'пятью',
				'6'  => 'шестью',
				'7'  => 'семью',
				'8'  => 'восемью',
				'9'  => 'девятью',
				'10' => 'десятью',
				'11' => 'одиннадцатью',
				'12' => 'двенадцатью',
				'13' => 'тринадцатью',
				'14' => 'четырнадцатью',
				'15' => 'пятнадцатью',
				'16' => 'шестнадцатью',
				'17' => 'семнадцатью',
				'18' => 'восемнадцатью',
				'19' => 'девятнадцатью',

				'20' => 'двадцатью',
				'30' => 'тридцатью',
				'40' => 'сорока',
				'50' => 'пятьюдесятью',
				'60' => 'шестьюдесятью',
				'70' => 'семьюдесятью',
				'80' => 'восемьюдесятью',
				'90' => 'девяноста',

				'100' => 'ста',
				'200' => 'двумястами',
				'300' => 'тремястами',
				'400' => 'четырьмястами',
				'500' => 'пятьюстами',
				'600' => 'шестьюстами',
				'700' => 'семьюстами',
				'800' => 'восемьюстами',
				'900' => 'девятьюстами',

				'1000' => 'одной тысячей',
				'2000' => 'двумя тысячами',
				'3000' => 'тремя тысячами',
				'4000' => 'четырьмя тысячами',
				'5000' => 'пятью тысячами',
				'10000' => 'десятью тысячами',
				'90000' => 'девяноста тысячами',
				'100000' => 'ста тысячами',
				'900000' => 'девятьюстами тысячами',

				'1000000' => 'одним миллионом',
				'2000000' => 'двумя миллионами',
				'3000000' => 'тремя миллионами',
				'4000000' => 'четырьмя миллионами',
				'5000000' => 'пятью миллионами',
				'10000000' => 'десятью миллионами',
				'90000000' => 'девяноста миллионами',
				'100000000' => 'ста миллионами',
				'900000000' => 'девятьюстами миллионами',

				'1000000000' => 'одним миллиардом',
				'2000000000' => 'двумя миллиардами',
				'3000000000' => 'тремя миллиардами',
				'4000000000' => 'четырьмя миллиардами',
				'5000000000' => 'пятью миллиардами',
				'10000000000' => 'десятью миллиардами',
				'90000000000' => 'девяноста миллиардами',
				'100000000000' => 'ста миллиардами',
				'900000000000' => 'девятьюстами миллиардами',

				'1000000000000' => 'одним триллионом',
				'2000000000000' => 'двумя триллионами',
				'3000000000000' => 'тремя триллионами',
				'4000000000000' => 'четырьмя триллионами',
				'5000000000000' => 'пятью триллионами',
				'10000000000000' => 'десятью триллионами',
				'90000000000000' => 'девяноста триллионами',
				'100000000000000' => 'ста триллионами',
				'900000000000000' => 'девятьюстами триллионами',

				'1002' => 'одной тысячей двумя',
				'1000003' => 'одним миллионом тремя',
				'1000000004' => 'одним миллиардом четырьмя',
				'1000000000005' => 'одним триллионом пятью',
				'2006000' => 'двумя миллионами шестью тысячами',
				'2000007000' => 'двумя миллиардами семью тысячами',
				'2000000002000' => 'двумя триллионами двумя тысячами',
				'5008000000' => 'пятью миллиардами восемью миллионами',
				'6000030000000' => 'шестью триллионами тридцатью миллионами',
				'1007000000000' => 'одним триллионом семью миллиардами',
				'930651342187216' => 'девятьюстами тридцатью триллионами шестьюстами пятьюдесятью одним миллиардом тремястами сорока двумя миллионами ста восемьюдесятью семью тысячами двумястами шестнадцатью',
				'999999999999999' => 'девятьюстами девяноста девятью триллионами девятьюстами девяноста девятью миллиардами девятьюстами девяноста девятью миллионами девятьюстами девяноста девятью тысячами девятьюстами девяноста девятью',

			},
			'feminine' => { # Женский род: 'одной', а остальные как для Мужского рода
				'0'  => 'нолём',	# like 'masculine'
				'1'  => 'одной',	###
				'2'  => 'двумя',	# like 'masculine'
				'3'  => 'тремя',	# ...
				'4'  => 'четырьмя',	# ...
				'5'  => 'пятью',	# ...
				'10' => 'десятью',	# ...

				'100' => 'ста',
				'900' => 'девятьюстами',

				'1000' => 'одной тысячей',
				'2000' => 'двумя тысячами',
				'90000' => 'девяноста тысячами',
				'100000' => 'ста тысячами',

				'1000000' => 'одним миллионом',
				'2000000' => 'двумя миллионами',
				'90000000' => 'девяноста миллионами',
				'100000000' => 'ста миллионами',

				'1000000000' => 'одним миллиардом',
				'2000000000' => 'двумя миллиардами',
				'90000000000' => 'девяноста миллиардами',
				'100000000000' => 'ста миллиардами',

				'1000000000000' => 'одним триллионом',
				'2000000000000' => 'двумя триллионами',
				'90000000000000' => 'девяноста триллионами',
				'100000000000000' => 'ста триллионами',

				'101101102101101' => 'ста одним триллионом ста одним миллиардом ста двумя миллионами ста одной тысячей ста одной',

#	'alternative' => [ # 0..19
#		'','одною' # остальные как для Мужского рода
#	],

			},
			'neuter' => { # Средний род: all like 'masculine'
				'0'  => 'нолём',
				'1'  => 'одним',
				'2'  => 'двумя',
				'3'  => 'тремя',
				'4'  => 'четырьмя',
				'5'  => 'пятью',
				'10' => 'десятью',

				'100' => 'ста',
				'900' => 'девятьюстами',

				'1000' => 'одной тысячей',
				'2000' => 'двумя тысячами',
				'90000' => 'девяноста тысячами',
				'100000' => 'ста тысячами',

				'1000000' => 'одним миллионом',
				'2000000' => 'двумя миллионами',
				'90000000' => 'девяноста миллионами',
				'100000000' => 'ста миллионами',

				'1000000000' => 'одним миллиардом',
				'2000000000' => 'двумя миллиардами',
				'90000000000' => 'девяноста миллиардами',
				'100000000000' => 'ста миллиардами',

				'1000000000000' => 'одним триллионом',
				'2000000000000' => 'двумя триллионами',
				'90000000000000' => 'девяноста триллионами',
				'100000000000000' => 'ста триллионами',

				'101101102101101' => 'ста одним триллионом ста одним миллиардом ста двумя миллионами ста одной тысячей ста одним',

			},

			'plural' => {	# множественное число
				'1' => {'inanimate' => 'одними'}, # неодушевлённый, animate(одушевлённый)
				'1000' => {'inanimate' => 'одними тысячами'},
				'1000000' => {'inanimate' => 'одними миллионами'},
				'1000000000' => {'inanimate' => 'одними миллиардами'},
				'1000000000000' => {'inanimate' => 'одними триллионами'},
			},

		}, #--------------------------------------------
		'p' => { # prepositional (Предложный падеж) : о ком? о чём?
			'masculine' => { # Мужской род
				'0'  => 'ноле',
				'1'  => 'одном',
				'2'  => 'двух',
				'3'  => 'трёх',
				'4'  => 'четырёх',
				'5'  => 'пяти',
				'6'  => 'шести',
				'7'  => 'семи',
				'8'  => 'восьми',
				'9'  => 'девяти',
				'10' => 'десяти',

				'11' => 'одиннадцати',
				'12' => 'двенадцати',
				'13' => 'тринадцати',
				'14' => 'четырнадцати',
				'15' => 'пятнадцати',
				'16' => 'шестнадцати',
				'17' => 'семнадцати',
				'18' => 'восемнадцати',
				'19' => 'девятнадцати',

				'20' => 'двадцати',
				'30' => 'тридцати',
				'40' => 'сорока',
				'50' => 'пятидесяти',
				'60' => 'шестидесяти',
				'70' => 'семидесяти',
				'80' => 'восьмидесяти',
				'90' => 'девяноста',

				'100' => 'ста',
				'200' => 'двухстах',
				'300' => 'трёхстах',
				'400' => 'четырёхстах',
				'500' => 'пятистах',
				'600' => 'шестистах',
				'700' => 'семистах',
				'800' => 'восьмистах',
				'900' => 'девятистах',

				'1000' => 'одной тысяче',
				'2000' => 'двух тысячах',
				'3000' => 'трёх тысячах',
				'4000' => 'четырёх тысячах',
				'5000' => 'пяти тысячах',
				'10000' => 'десяти тысячах',
				'90000' => 'девяноста тысячах',
				'100000' => 'ста тысячах',
				'900000' => 'девятистах тысячах',

				'1000000' => 'одном миллионе',
				'2000000' => 'двух миллионах',
				'3000000' => 'трёх миллионах',
				'4000000' => 'четырёх миллионах',
				'5000000' => 'пяти миллионах',
				'10000000' => 'десяти миллионах',
				'90000000' => 'девяноста миллионах',
				'100000000' => 'ста миллионах',
				'900000000' => 'девятистах миллионах',

				'1000000000' => 'одном миллиарде',
				'2000000000' => 'двух миллиардах',
				'3000000000' => 'трёх миллиардах',
				'4000000000' => 'четырёх миллиардах',
				'5000000000' => 'пяти миллиардах',
				'10000000000' => 'десяти миллиардах',
				'90000000000' => 'девяноста миллиардах',
				'100000000000' => 'ста миллиардах',
				'900000000000' => 'девятистах миллиардах',

				'1000000000000' => 'одном триллионе',
				'2000000000000' => 'двух триллионах',
				'3000000000000' => 'трёх триллионах',
				'4000000000000' => 'четырёх триллионах',
				'5000000000000' => 'пяти триллионах',
				'10000000000000' => 'десяти триллионах',
				'90000000000000' => 'девяноста триллионах',
				'100000000000000' => 'ста триллионах',
				'900000000000000' => 'девятистах триллионах',

				'1002' => 'одной тысяче двух',
				'1000003' => 'одном миллионе трёх',
				'1000000004' => 'одном миллиарде четырёх',
				'1000000000005' => 'одном триллионе пяти',
				'2006000' => 'двух миллионах шести тысячах',
				'2000007000' => 'двух миллиардах семи тысячах',
				'2000000002000' => 'двух триллионах двух тысячах',
				'5008000000' => 'пяти миллиардах восьми миллионах',
				'6000030000000' => 'шести триллионах тридцати миллионах',
				'1007000000000' => 'одном триллионе семи миллиардах',
				'930651342187216' => 'девятистах тридцати триллионах шестистах пятидесяти одном миллиарде трёхстах сорока двух миллионах ста восьмидесяти семи тысячах двухстах шестнадцати',
				'999999999999999' => 'девятистах девяноста девяти триллионах девятистах девяноста девяти миллиардах девятистах девяноста девяти миллионах девятистах девяноста девяти тысячах девятистах девяноста девяти',

			},
			'feminine' => { # Женский род: 'одной', а остальные как для Мужского рода
				'0'  => 'ноле',	# like 'masculine'
				'1'  => 'одной',	###
				'2'  => 'двух',	# like 'masculine'
				'3'  => 'трёх',	# ...
				'4'  => 'четырёх',	# ...
				'5'  => 'пяти',	# ...
				'10' => 'десяти',	# ...

				'100' => 'ста',
				'900' => 'девятистах',

				'1000' => 'одной тысяче',
				'2000' => 'двух тысячах',
				'90000' => 'девяноста тысячах',
				'100000' => 'ста тысячах',

				'1000000' => 'одном миллионе',
				'2000000' => 'двух миллионах',
				'90000000' => 'девяноста миллионах',
				'100000000' => 'ста миллионах',

				'1000000000' => 'одном миллиарде',
				'2000000000' => 'двух миллиардах',
				'90000000000' => 'девяноста миллиардах',
				'100000000000' => 'ста миллиардах',

				'1000000000000' => 'одном триллионе',
				'2000000000000' => 'двух триллионах',
				'90000000000000' => 'девяноста триллионах',
				'100000000000000' => 'ста триллионах',

				'101101102101101' => 'ста одном триллионе ста одном миллиарде ста двух миллионах ста одной тысяче ста одной',

			},
			'neuter' => { # Средний род: all like 'masculine'
				'0'  => 'ноле',
				'1'  => 'одном',
				'2'  => 'двух',
				'3'  => 'трёх',
				'4'  => 'четырёх',
				'5'  => 'пяти',
				'10' => 'десяти',

				'100' => 'ста',
				'900' => 'девятистах',

				'1000' => 'одной тысяче',
				'2000' => 'двух тысячах',
				'90000' => 'девяноста тысячах',
				'100000' => 'ста тысячах',

				'1000000' => 'одном миллионе',
				'2000000' => 'двух миллионах',
				'90000000' => 'девяноста миллионах',
				'100000000' => 'ста миллионах',

				'1000000000' => 'одном миллиарде',
				'2000000000' => 'двух миллиардах',
				'90000000000' => 'девяноста миллиардах',
				'100000000000' => 'ста миллиардах',

				'1000000000000' => 'одном триллионе',
				'2000000000000' => 'двух триллионах',
				'90000000000000' => 'девяноста триллионах',
				'100000000000000' => 'ста триллионах',

				'101101102101101' => 'ста одном триллионе ста одном миллиарде ста двух миллионах ста одной тысяче ста одном',

			},

			'plural' => {	# prepositional (Предложный падеж): множественное число
				'1' => {'inanimate' => 'одних'}, # неодушевлённый и одушевлённый
				'1000' => {'inanimate' => 'одних тысячах'},
				'1000000' => {'inanimate' => 'одних миллионах'},
				'1000000000' => {'inanimate' => 'одних миллиардах'},
				'1000000000000' => {'inanimate' => 'одних триллионах'},
			},

		},

	}
}


sub datest_prolog {

	return {
		'g' => { # genitive (Родительный падеж): кого? чего?
			'prolog' => ['в'],
			'masculine' => { # Мужской род
				'1'  => 'в одного',
				'2'  => 'в двух',
				'3'  => 'в трёх',
				'4'  => 'в четырёх',
				'5'  => 'в пяти',
				'100' => 'в ста',
				'1000' => 'в одной тысячи',
				'1000000' => 'в одного миллиона',
			},
			'feminine' => { # Женский род
				'1'  => 'в одной',
				'2'  => 'в двух',	# like 'masculine'
			},
			'neuter' => { # Средний род
				'4'  => 'в четырёх',	# ...
			},

			'plural' => {	# множественное число
				'1' => 'в одних',
			},

		}, #--------------------------------------------
		'i' => { # instrumental (Творительный падеж) : кем? чем?
			'prolog' => 'с', # ['с','со=с'],
			'masculine' => { # Мужской род, default
				'0'  => 'с нолём',
				'1'  => 'с одним',
				'2'  => 'с двумя',
				'3'  => 'с тремя',
				'4'  => 'с четырьмя',
				'5'  => 'с пятью',
				'7'  => 'с семью',

				'10' => 'с десятью',
				'11' => 'с одиннадцатью',
				'12' => 'с двенадцатью',
				'17'  => 'с семнадцатью',

				'20' => 'с двадцатью',
				'40' => 'с сорока',
				'70' => 'с семьюдесятью',
				'100' => 'со ста',
				'700' => 'с семьюстами',

				'1000' => 'с одной тысячей',
				'2000' => 'с двумя тысячами',
				'100000' => 'со ста тысячами',

				'1000000' => 'с одним миллионом',
				'2000000' => 'с двумя миллионами',
				'100000000' => 'со ста миллионами',

				'1000000000' => 'с одним миллиардом',
				'2000000000' => 'с двумя миллиардами',
				'100000000000' => 'со ста миллиардами',

				'1000000000000' => 'с одним триллионом',
				'2000000000000' => 'с двумя триллионами',
				'100000000000000' => 'со ста триллионами',

				'130651342187216' => 'со ста тридцатью триллионами шестьюстами пятьюдесятью одним миллиардом тремястами сорока двумя миллионами ста восемьюдесятью семью тысячами двумястами шестнадцатью',

			},
			'feminine' => { # Женский род: 'одной', а остальные как для Мужского рода
				'0'  => 'с нолём',	# like 'masculine'
				'1'  => 'с одной',	###
				'1000' => 'с одной тысячей',
			},

			'plural' => {	# множественное число
				'1' => 'с одними',
				'1000' => 'с одними тысячами',
				'1000000' => 'с одними миллионами',
				'1000000000' => 'с одними миллиардами',
				'1000000000000' => 'с одними триллионами',
			},

		}, #--------------------------------------------
		'p' => { # prepositional (Предложный падеж) : о ком? о чём?
			'prolog' => 'о', # ['о','об=од'],
			'masculine' => { # Мужской род
				'0'  => 'о ноле',
				'1'  => 'об одном',
				'2'  => 'о двух',
				'3'  => 'о трёх',
				'4'  => 'о четырёх',
				'5'  => 'о пяти',

				'10' => 'о десяти',
				'11' => 'об одиннадцати',
				'12' => 'о двенадцати',

				'20' => 'о двадцати',
				'100' => 'о ста',

				'1000' => 'об одной тысяче',
				'2000' => 'о двух тысячах',
				'11000' => 'об одиннадцати тысячах',
				'100000' => 'о ста тысячах',

				'1000000' => 'об одном миллионе',
				'2000000' => 'о двух миллионах',
				'11000000' => 'об одиннадцати миллионах',
				'100000000' => 'о ста миллионах',

				'1000000000' => 'об одном миллиарде',
				'2000000000' => 'о двух миллиардах',

				'1000000000000' => 'об одном триллионе',
				'2000000000000' => 'о двух триллионах',

				'1002' => 'об одной тысяче двух',
				'930651342187216' => 'о девятистах тридцати триллионах шестистах пятидесяти одном миллиарде трёхстах сорока двух миллионах ста восьмидесяти семи тысячах двухстах шестнадцати',

			},
			'feminine' => { # Женский род: 'одной', а остальные как для Мужского рода
				'1'  => 'об одной',	###
				'1000' => 'об одной тысяче',
				'2000' => 'о двух тысячах',
				'11000' => 'об одиннадцати тысячах',
			},

			'plural' => {	# prepositional (Предложный падеж): множественное число
				'1' => 'об одних', # неодушевлённый и одушевлённый
				'1000' => 'об одних тысячах',
				'1000000' => 'об одних миллионах',
			},

		},

	}

}


sub datest_epilog {
	return {
		'RUB' => {	# Российский рубль
			'n' => { # nominative (Именительный падеж): есть кто? что?
				'0'  => 'ноль рублей',
				'1'  => 'один рубль',
				'2'  => 'два рубля',
				'3'  => 'три рубля',
				'4'  => 'четыре рубля',
				'5'  => 'пять рублей',
				'6'  => 'шесть рублей',
				'7'  => 'семь рублей',
				'8'  => 'восемь рублей',
				'9'  => 'девять рублей',
				'10' => 'десять рублей',
				'11' => 'одиннадцать рублей',

				'20' => 'двадцать рублей',
				'30' => 'тридцать рублей',
				'40' => 'сорок рублей',
				'50' => 'пятьдесят рублей',
				'90' => 'девяносто рублей',

				'100' => 'сто рублей',
				'1000' => 'одна тысяча рублей',
				'2000' => 'две тысячи рублей',

				'1000000' => 'один миллион рублей',
				'2000000' => 'два миллиона рублей',

				'1000000000' => 'один миллиард рублей',
				'2000000000' => 'два миллиарда рублей',

				'1000000000000' => 'один триллион рублей',
				'2000000000000' => 'два триллиона рублей',

				'1002' => 'одна тысяча два рубля',
				'1000000000005' => 'один триллион пять рублей',

				'930651342187216' => 'девятьсот тридцать триллионов шестьсот пятьдесят один миллиард триста сорок два миллиона сто восемьдесят семь тысяч двести шестнадцать рублей',

				'plural' => {	# множественное число
					'1'  => 'одни рубли',
					'1000' => 'одни тысячи рублей',	# !
					'1000000' => 'одни миллионы рублей',	# !
					'1000000000' => 'одни миллиарды рублей',	# !
					'1000000000000' => 'одни триллионы рублей',	# !
				},

			},
			'g' => { # genitive (Родительный падеж): нет кого? чего?
				'0'  => 'ноля рублей',
				'1'  => 'одного рубля',
				'2'  => 'двух рублей',
				'5'  => 'пяти рублей',
				'10' => 'десяти рублей',
				'11' => 'одиннадцати рублей',
				'100' => 'ста рублей',
				'1000' => 'одной тысячи рублей',
				'2000' => 'двух тысяч рублей',

				'1000000' => 'одного миллиона рублей',
				'2000000' => 'двух миллионов рублей',

				'1000000000' => 'одного миллиарда рублей',
				'2000000000' => 'двух миллиардов рублей',

				'1000000000000' => 'одного триллиона рублей',
				'2000000000000' => 'двух триллионов рублей',

				'1002' => 'одной тысячи двух рублей',
				'1000000000005' => 'одного триллиона пяти рублей',

				'930651342187216' => 'девятисот тридцати триллионов шестисот пятидесяти одного миллиарда трёхсот сорока двух миллионов ста восьмидесяти семи тысяч двухсот шестнадцати рублей',

				'plural' => {	# множественное число
					'1' => 'одних рублей',
					'1000' => 'одних тысяч рублей',
					'1000000' => 'одних миллионов рублей',
					'1000000000' => 'одних миллиардов рублей',
					'1000000000000' => 'одних триллионов рублей',
				},

			},
			'd' => { # dative (Дательный падеж): рад кому? чему?
				'0'  => 'нолю рублям',
				'1'  => 'одному рублю',
				'2'  => 'двум рублям',
				'5'  => 'пяти рублям',
				'10' => 'десяти рублям',
				'11' => 'одиннадцати рублям',
				'100' => 'ста рублям',
				'1000' => 'одной тысяче рублей',	# ям ?
				'2000' => 'двум тысячам рублей',	# ям ?

				'1000000' => 'одному миллиону рублей',	# ям ?
				'2000000' => 'двум миллионам рублей',	# ям ?

				'1000000000' => 'одному миллиарду рублей',	# ям ?
				'2000000000' => 'двум миллиардам рублей',	# ям ?

				'1000000000000' => 'одному триллиону рублей',	# ям ?
				'2000000000000' => 'двум триллионам рублей',	# ям ?

				'1002' => 'одной тысяче двум рублям',
				'1000000000005' => 'одному триллиону пяти рублям',

				'930651342187216' => 'девятистам тридцати триллионам шестистам пятидесяти одному миллиарду трёхстам сорока двум миллионам ста восьмидесяти семи тысячам двумстам шестнадцати рублям',

				'plural' => {	# множественное число
					'1' => 'одним рублям',
					'1000' => 'одним тысячам рублей',	# ям ?
					'1000000' => 'одним миллионам рублей',	# ям ?
					'1000000000' => 'одним миллиардам рублей',	# ям ?
					'1000000000000' => 'одним триллионам рублей',	# ям ?
				},

			},
			'a' => { # accusative (Винительный падеж): inanimate (неодушевлённый объект) вижу кого? что?
				'0'  => 'ноль рублей',	# 0..19 : неодушевлённый объект, как Именительный падеж
				'1'  => 'один рубль',
				'2'  => 'два рубля',
				'5'  => 'пять рублей',
				'10' => 'десять рублей',
				'11' => 'одиннадцать рублей',
				'100' => 'сто рублей',
				'1000' => 'одну тысячу рублей',	# !!!
				'2000' => 'две тысячи рублей',

				'1000000' => 'один миллион рублей',
				'2000000' => 'два миллиона рублей',

				'1000000000' => 'один миллиард рублей',
				'2000000000' => 'два миллиарда рублей',

				'1000000000000' => 'один триллион рублей',
				'2000000000000' => 'два триллиона рублей',

				'1002' => 'одну тысячу два рубля',
				'1000000000005' => 'один триллион пять рублей',

				'930651342187216' => 'девятьсот тридцать триллионов шестьсот пятьдесят один миллиард триста сорок два миллиона сто восемьдесят семь тысяч двести шестнадцать рублей',

				'plural' => {	# множественное число
					'1'  => 'одни рубли',
					'1000' => 'одни тысячи рублей',	# !
					'1000000' => 'одни миллионы рублей',	# !
					'1000000000' => 'одни миллиарды рублей',	# !
					'1000000000000' => 'одни триллионы рублей',	# !
				},

			},
			'i' => { # instrumental (Творительный падеж) : оплачу кем? чем?
				'0'  => 'нолём рублей',	# ями ?
				'1'  => 'одним рублём',
				'2'  => 'двумя рублями',
				'5'  => 'пятью рублями',
				'10' => 'десятью рублями',
				'11' => 'одиннадцатью рублями',
				'100' => 'ста рублями',
				'1000' => 'одной тысячей рублей',	# ями ?
				'2000' => 'двумя тысячами рублей',	# ями ?

				'1000000' => 'одним миллионом рублей',	# ями ?
				'2000000' => 'двумя миллионами рублей',	# ями ?

				'1000000000' => 'одним миллиардом рублей',	# ями ?
				'2000000000' => 'двумя миллиардами рублей',	# ями ?

				'1000000000000' => 'одним триллионом рублей',	# ями ?
				'2000000000000' => 'двумя триллионами рублей',	# ями ?

				'1002' => 'одной тысячей двумя рублями',
				'1000000000005' => 'одним триллионом пятью рублями',

				'930651342187216' => 'девятьюстами тридцатью триллионами шестьюстами пятьюдесятью одним миллиардом тремястами сорока двумя миллионами ста восемьюдесятью семью тысячами двумястами шестнадцатью рублями',

				'plural' => {	# множественное число
					'1' => 'одними рублями',
					'1000' => 'одними тысячами рублей',	# ями ?
					'1000000' => 'одними миллионами рублей',	# ями ?
					'1000000000' => 'одними миллиардами рублей',	# ями ?
					'1000000000000' => 'одними триллионами рублей',	# ями ?
				},

			},
			'p' => { # prepositional (Предложный падеж) : думаю о ком? о чём?
				'0'  => 'ноле рублей',	# ях ?
				'1'  => 'одном рубле',
				'2'  => 'двух рублях',
				'5'  => 'пяти рублях',
				'10' => 'десяти рублях',
				'11' => 'одиннадцати рублях',
				'100' => 'ста рублях',

				'1000' => 'одной тысяче рублей',	# ях ?
				'2000' => 'двух тысячах рублей',	# ях ?

				'1000000' => 'одном миллионе рублей',	# ях ?
				'2000000' => 'двух миллионах рублей',	# ях ?

				'1000000000' => 'одном миллиарде рублей',	# ях ?
				'2000000000' => 'двух миллиардах рублей',	# ях ?

				'1000000000000' => 'одном триллионе рублей',	# ях ?
				'2000000000000' => 'двух триллионах рублей',	# ях ?

				'1002' => 'одной тысяче двух рублях',
				'1000000000005' => 'одном триллионе пяти рублях',

				'930651342187216' => 'девятистах тридцати триллионах шестистах пятидесяти одном миллиарде трёхстах сорока двух миллионах ста восьмидесяти семи тысячах двухстах шестнадцати рублях',

				'plural' => {	# множественное число
					'1' => 'одних рублях',
					'1000' => 'одних тысячах рублей',	# ях ?
					'1000000' => 'одних миллионах рублей',	# ях ?
					'1000000000' => 'одних миллиардах рублей',	# ях ?
					'1000000000000' => 'одних триллионах рублей',	# ях ?
				},
			},

		},
		'rub' => {	# Российская копейка
			'n' => { # nominative (Именительный падеж): есть кто? что?
				'0'  => 'ноль копеек',
				'1'  => 'одна копейка',
				'2'  => 'две копейки',
				'3'  => 'три копейки',
				'4'  => 'четыре копейки',
				'5'  => 'пять копеек',
				'6'  => 'шесть копеек',
				'7'  => 'семь копеек',
				'8'  => 'восемь копеек',
				'9'  => 'девять копеек',
				'10' => 'десять копеек',
				'11' => 'одиннадцать копеек',

				'20' => 'двадцать копеек',
				'30' => 'тридцать копеек',
				'40' => 'сорок копеек',
				'50' => 'пятьдесят копеек',
				'90' => 'девяносто копеек',

				'100' => 'сто копеек',
				'1000' => 'одна тысяча копеек',
				'2000' => 'две тысячи копеек',

				'1000000' => 'один миллион копеек',
				'2000000' => 'два миллиона копеек',

				'1000000000' => 'один миллиард копеек',
				'2000000000' => 'два миллиарда копеек',

				'1000000000000' => 'один триллион копеек',
				'2000000000000' => 'два триллиона копеек',

				'1002' => 'одна тысяча две копейки',
				'1000000000005' => 'один триллион пять копеек',

				'930651342187216' => 'девятьсот тридцать триллионов шестьсот пятьдесят один миллиард триста сорок два миллиона сто восемьдесят семь тысяч двести шестнадцать копеек',

				'plural' => {	# множественное число
					'1'  => 'одни копейки',
					'1000' => 'одни тысячи копеек',
					'1000000' => 'одни миллионы копеек',
					'1000000000' => 'одни миллиарды копеек',
					'1000000000000' => 'одни триллионы копеек',
				},

			},
			'g' => { # genitive (Родительный падеж): нет кого? чего?
				'0'  => 'ноля копеек',
				'1'  => 'одной копейки',
				'2'  => 'двух копеек',
				'5'  => 'пяти копеек',
				'10' => 'десяти копеек',
				'11' => 'одиннадцати копеек',
				'100' => 'ста копеек',
				'1000' => 'одной тысячи копеек',
				'2000' => 'двух тысяч копеек',

				'1000000' => 'одного миллиона копеек',
				'2000000' => 'двух миллионов копеек',

				'1000000000' => 'одного миллиарда копеек',
				'2000000000' => 'двух миллиардов копеек',

				'1000000000000' => 'одного триллиона копеек',
				'2000000000000' => 'двух триллионов копеек',

				'1002' => 'одной тысячи двух копеек',
				'1000000000005' => 'одного триллиона пяти копеек',

				'930651342187216' => 'девятисот тридцати триллионов шестисот пятидесяти одного миллиарда трёхсот сорока двух миллионов ста восьмидесяти семи тысяч двухсот шестнадцати копеек',

				'plural' => {	# множественное число
					'1' => 'одних копеек',
					'1000' => 'одних тысяч копеек',
					'1000000' => 'одних миллионов копеек',
					'1000000000' => 'одних миллиардов копеек',
					'1000000000000' => 'одних триллионов копеек',
				},

			},
			'd' => { # dative (Дательный падеж): рад кому? чему?
				'0'  => 'нолю копеек',	# йкам ?
				'1'  => 'одной копейке',
				'2'  => 'двум копейкам',
				'5'  => 'пяти копейкам',
				'10' => 'десяти копейкам',
				'11' => 'одиннадцати копейкам',
				'100' => 'ста копейкам',
				'1000' => 'одной тысяче копеек',	# йкам ?
				'2000' => 'двум тысячам копеек',	# йкам ?

				'1000000' => 'одному миллиону копеек',	# йкам ?
				'2000000' => 'двум миллионам копеек',	# йкам ?

				'1000000000' => 'одному миллиарду копеек',	# йкам ?
				'2000000000' => 'двум миллиардам копеек',	# йкам ?

				'1000000000000' => 'одному триллиону копеек',	# йкам ?
				'2000000000000' => 'двум триллионам копеек',	# йкам ?

				'1002' => 'одной тысяче двум копейкам',
				'1000000000005' => 'одному триллиону пяти копейкам',

				'930651342187216' => 'девятистам тридцати триллионам шестистам пятидесяти одному миллиарду трёхстам сорока двум миллионам ста восьмидесяти семи тысячам двумстам шестнадцати копейкам',

				'plural' => {	# множественное число
					'1' => 'одним копейкам',
					'1000' => 'одним тысячам копеек',
					'1000000' => 'одним миллионам копеек',
					'1000000000' => 'одним миллиардам копеек',
					'1000000000000' => 'одним триллионам копеек',
				},

			},
			'a' => { # accusative (Винительный падеж): inanimate (неодушевлённый объект) вижу кого? что?
				'0'  => 'ноль копеек',	# 0..19 : неодушевлённый объект, как Именительный падеж
				'1'  => 'одну копейку',
				'2'  => 'две копейки',
				'5'  => 'пять копеек',
				'10' => 'десять копеек',
				'11' => 'одиннадцать копеек',
				'100' => 'сто копеек',
				'1000' => 'одну тысячу копеек',	# !!!
				'2000' => 'две тысячи копеек',

				'1000000' => 'один миллион копеек',
				'2000000' => 'два миллиона копеек',

				'1000000000' => 'один миллиард копеек',
				'2000000000' => 'два миллиарда копеек',

				'1000000000000' => 'один триллион копеек',
				'2000000000000' => 'два триллиона копеек',

				'1002' => 'одну тысячу две копейки',
				'1000000000005' => 'один триллион пять копеек',

				'930651342187216' => 'девятьсот тридцать триллионов шестьсот пятьдесят один миллиард триста сорок два миллиона сто восемьдесят семь тысяч двести шестнадцать копеек',

				'plural' => {	# множественное число
					'1'  => 'одни копейки',
					'1000' => 'одни тысячи копеек',
					'1000000' => 'одни миллионы копеек',
					'1000000000' => 'одни миллиарды копеек',
					'1000000000000' => 'одни триллионы копеек',
				},

			},
			'i' => { # instrumental (Творительный падеж) : оплачу кем? чем?
				'0'  => 'нолём копеек',	# йками ?
				'1'  => 'одной копейкой',
				'2'  => 'двумя копейками',
				'5'  => 'пятью копейками',
				'10' => 'десятью копейками',
				'11' => 'одиннадцатью копейками',
				'100' => 'ста копейками',
				'1000' => 'одной тысячей копеек',	# йками ?
				'2000' => 'двумя тысячами копеек',	# йками ?

				'1000000' => 'одним миллионом копеек',	# йками ?
				'2000000' => 'двумя миллионами копеек',	# йками ?

				'1000000000' => 'одним миллиардом копеек',	# йками ?
				'2000000000' => 'двумя миллиардами копеек',	# йками ?

				'1000000000000' => 'одним триллионом копеек',	# йками ?
				'2000000000000' => 'двумя триллионами копеек',	# йками ?

				'1002' => 'одной тысячей двумя копейками',
				'1000000000005' => 'одним триллионом пятью копейками',

				'930651342187216' => 'девятьюстами тридцатью триллионами шестьюстами пятьюдесятью одним миллиардом тремястами сорока двумя миллионами ста восемьюдесятью семью тысячами двумястами шестнадцатью копейками',

				'plural' => {	# множественное число
					'1' => 'одними копейками',
					'100' => 'ста копейками',
					'1000' => 'одними тысячами копеек',	# йками ?
					'1000000' => 'одними миллионами копеек',	# йками ?
					'1000000000' => 'одними миллиардами копеек',	# йками ?
					'1000000000000' => 'одними триллионами копеек',	# йками ?
				},

			},
			'p' => { # prepositional (Предложный падеж) : думаю о ком? о чём?
				'0'  => 'ноле копеек',	# йках ?
				'1'  => 'одной копейке',
				'2'  => 'двух копейках',
				'5'  => 'пяти копейках',
				'10' => 'десяти копейках',
				'11' => 'одиннадцати копейках',
				'100' => 'ста копейках',

				'1000' => 'одной тысяче копеек',	# йках ?
				'2000' => 'двух тысячах копеек',	# йках ?

				'1000000' => 'одном миллионе копеек',	# йках ?
				'2000000' => 'двух миллионах копеек',	# йках ?

				'1000000000' => 'одном миллиарде копеек',	# йках ?
				'2000000000' => 'двух миллиардах копеек',	# йках ?

				'1000000000000' => 'одном триллионе копеек',	# йках ?
				'2000000000000' => 'двух триллионах копеек',	# йках ?

				'1002' => 'одной тысяче двух копейках',
				'1000000000005' => 'одном триллионе пяти копейках',

				'930651342187216' => 'девятистах тридцати триллионах шестистах пятидесяти одном миллиарде трёхстах сорока двух миллионах ста восьмидесяти семи тысячах двухстах шестнадцати копейках',

				'plural' => {	# множественное число
					'1' => 'одних копейках',
					'100' => 'ста копейках',
					'1000' => 'одних тысячах копеек',	# йках ?
					'1000000' => 'одних миллионах копеек',	# йках ?
					'1000000000' => 'одних миллиардах копеек',	# йках ?
					'1000000000000' => 'одних триллионах копеек',	# йках ?
				},
			},

		},
		'BYR' => {	# Белорусский рубль
			'n' => { # nominative (Именительный падеж): есть кто? что?
				'0'  => 'ноль белорусских рублей',
				'1'  => 'один белорусский рубль',
				'2'  => 'два белорусских рубля',
				'3'  => 'три белорусских рубля',
				'4'  => 'четыре белорусских рубля',
				'5'  => 'пять белорусских рублей',
				'6'  => 'шесть белорусских рублей',

				'100' => 'сто белорусских рублей',
				'1000' => 'одна тысяча белорусских рублей',
				'1000000' => 'один миллион белорусских рублей',
				'1000000000' => 'один миллиард белорусских рублей',
				'1000000000000' => 'один триллион белорусских рублей',

				'plural' => {	# множественное число
					'1'  => 'одни белорусские рубли',
					'1000' => 'одни тысячи белорусских рублей',
					'1000000' => 'одни миллионы белорусских рублей',
					'1000000000' => 'одни миллиарды белорусских рублей',
					'1000000000000' => 'одни триллионы белорусских рублей',
				},

			},
			'g' => { # genitive (Родительный падеж): нет кого? чего?
				'0'  => 'ноля белорусских рублей',
				'1'  => 'одного белорусского рубля',
				'2'  => 'двух белорусских рублей',
				'5'  => 'пяти белорусских рублей',
				'10' => 'десяти белорусских рублей',
				'11' => 'одиннадцати белорусских рублей',
				'100' => 'ста белорусских рублей',
				'1000' => 'одной тысячи белорусских рублей',
				'1000000' => 'одного миллиона белорусских рублей',
				'1000000000' => 'одного миллиарда белорусских рублей',
				'1000000000000' => 'одного триллиона белорусских рублей',

				'plural' => {	# множественное число
					'1' => 'одних белорусских рублей',
					'1000' => 'одних тысяч белорусских рублей',
					'1000000' => 'одних миллионов белорусских рублей',
					'1000000000' => 'одних миллиардов белорусских рублей',
					'1000000000000' => 'одних триллионов белорусских рублей',
				},

			},
			'd' => { # dative (Дательный падеж): рад кому? чему?
				'0'  => 'нолю белорусским рублям',
				'1'  => 'одному белорусскому рублю',
				'2'  => 'двум белорусским рублям',
				'5'  => 'пяти белорусским рублям',
				'10' => 'десяти белорусским рублям',
				'11' => 'одиннадцати белорусским рублям',
				'100' => 'ста белорусским рублям',
				'1000' => 'одной тысяче белорусских рублей',
				'1000000' => 'одному миллиону белорусских рублей',
				'1000000000' => 'одному миллиарду белорусских рублей',
				'1000000000000' => 'одному триллиону белорусских рублей',

				'plural' => {	# множественное число
					'1' => 'одним белорусским рублям',
					'1000' => 'одним тысячам белорусских рублей',
					'1000000' => 'одним миллионам белорусских рублей',
					'1000000000' => 'одним миллиардам белорусских рублей',
					'1000000000000' => 'одним триллионам белорусских рублей',
				},

			},
			'a' => { # accusative (Винительный падеж): inanimate (неодушевлённый объект) вижу кого? что?
				'0'  => 'ноль белорусских рублей',	# 0..19 : неодушевлённый объект, как Именительный падеж
				'1'  => 'один белорусский рубль',
				'2'  => 'два белорусских рубля',
				'5'  => 'пять белорусских рублей',
				'10' => 'десять белорусских рублей',
				'11' => 'одиннадцать белорусских рублей',
				'100' => 'сто белорусских рублей',
				'1000' => 'одну тысячу белорусских рублей',
				'1000000' => 'один миллион белорусских рублей',
				'1000000000' => 'один миллиард белорусских рублей',
				'1000000000000' => 'один триллион белорусских рублей',

				'plural' => {	# множественное число
					'1'  => 'одни белорусские рубли',
					'1000' => 'одни тысячи белорусских рублей',
					'1000000' => 'одни миллионы белорусских рублей',
					'1000000000' => 'одни миллиарды белорусских рублей',
					'1000000000000' => 'одни триллионы белорусских рублей',
				},

			},
			'i' => { # instrumental (Творительный падеж) : оплачу кем? чем?
				'0'  => 'нолём белорусских рублей',
				'1'  => 'одним белорусским рублём',
				'2'  => 'двумя белорусскими рублями',
				'5'  => 'пятью белорусскими рублями',
				'10' => 'десятью белорусскими рублями',
				'11' => 'одиннадцатью белорусскими рублями',
				'100' => 'ста белорусскими рублями',
				'1000' => 'одной тысячей белорусских рублей',
				'1000000' => 'одним миллионом белорусских рублей',
				'1000000000' => 'одним миллиардом белорусских рублей',
				'1000000000000' => 'одним триллионом белорусских рублей',

				'plural' => {	# множественное число
					'1' => 'одними белорусскими рублями',
					'1000' => 'одними тысячами белорусских рублей',
					'1000000' => 'одними миллионами белорусских рублей',
					'1000000000' => 'одними миллиардами белорусских рублей',
					'1000000000000' => 'одними триллионами белорусских рублей',
				},

			},
			'p' => { # prepositional (Предложный падеж) : думаю о ком? о чём?
				'0'  => 'ноле белорусских рублей',
				'1'  => 'одном белорусском рубле',
				'2'  => 'двух белорусских рублях',
				'5'  => 'пяти белорусских рублях',
				'10' => 'десяти белорусских рублях',
				'11' => 'одиннадцати белорусских рублях',
				'100' => 'ста белорусских рублях',
				'1000' => 'одной тысяче белорусских рублей',
				'1000000' => 'одном миллионе белорусских рублей',
				'1000000000' => 'одном миллиарде белорусских рублей',
				'1000000000000' => 'одном триллионе белорусских рублей',

				'plural' => {	# множественное число
					'1' => 'одних белорусских рублях',
					'1000' => 'одних тысячах белорусских рублей',
					'1000000' => 'одних миллионах белорусских рублей',
					'1000000000' => 'одних миллиардах белорусских рублей',
					'1000000000000' => 'одних триллионах белорусских рублей',
				},
			},

		},
		'840' => {	# United States dollar
			'n' => { # nominative (Именительный падеж): есть кто? что?
				'0'  => 'ноль долларов',
				'1'  => 'один доллар',
				'2'  => 'два доллара',
				'3'  => 'три доллара',
				'4'  => 'четыре доллара',
				'5'  => 'пять долларов',
				'6'  => 'шесть долларов',
				'7'  => 'семь долларов',
				'8'  => 'восемь долларов',
				'9'  => 'девять долларов',
				'10' => 'десять долларов',
				'11' => 'одиннадцать долларов',

				'20' => 'двадцать долларов',
				'30' => 'тридцать долларов',
				'40' => 'сорок долларов',
				'50' => 'пятьдесят долларов',
				'90' => 'девяносто долларов',

				'100' => 'сто долларов',
				'1000' => 'одна тысяча долларов',
				'2000' => 'две тысячи долларов',

				'1000000' => 'один миллион долларов',
				'2000000' => 'два миллиона долларов',

				'1000000000' => 'один миллиард долларов',
				'2000000000' => 'два миллиарда долларов',

				'1000000000000' => 'один триллион долларов',
				'2000000000000' => 'два триллиона долларов',

				'1002' => 'одна тысяча два доллара',
				'1000000000005' => 'один триллион пять долларов',

				'930651342187216' => 'девятьсот тридцать триллионов шестьсот пятьдесят один миллиард триста сорок два миллиона сто восемьдесят семь тысяч двести шестнадцать долларов',

				'plural' => {	# множественное число
					'1'  => 'одни доллары',
					'1000' => 'одни тысячи долларов',
					'1000000' => 'одни миллионы долларов',
					'1000000000' => 'одни миллиарды долларов',
					'1000000000000' => 'одни триллионы долларов',
				},

			},
			'g' => { # genitive (Родительный падеж): нет кого? чего?
				'0'  => 'ноля долларов',
				'1'  => 'одного доллара',
				'2'  => 'двух долларов',
				'5'  => 'пяти долларов',
				'10' => 'десяти долларов',
				'11' => 'одиннадцати долларов',
				'100' => 'ста долларов',
				'1000' => 'одной тысячи долларов',
				'2000' => 'двух тысяч долларов',

				'1000000' => 'одного миллиона долларов',
				'2000000' => 'двух миллионов долларов',

				'1000000000' => 'одного миллиарда долларов',
				'2000000000' => 'двух миллиардов долларов',

				'1000000000000' => 'одного триллиона долларов',
				'2000000000000' => 'двух триллионов долларов',

				'1002' => 'одной тысячи двух долларов',
				'1000000000005' => 'одного триллиона пяти долларов',

				'930651342187216' => 'девятисот тридцати триллионов шестисот пятидесяти одного миллиарда трёхсот сорока двух миллионов ста восьмидесяти семи тысяч двухсот шестнадцати долларов',

				'plural' => {	# множественное число
					'1' => 'одних долларов',
					'1000' => 'одних тысяч долларов',
					'1000000' => 'одних миллионов долларов',
					'1000000000' => 'одних миллиардов долларов',
					'1000000000000' => 'одних триллионов долларов',
				},

			},
			'd' => { # dative (Дательный падеж): рад кому? чему?
				'0'  => 'нолю долларам',	# ов ??? 
				'1'  => 'одному доллару',
				'2'  => 'двум долларам',
				'5'  => 'пяти долларам',
				'10' => 'десяти долларам',
				'11' => 'одиннадцати долларам',
				'100' => 'ста долларам',
				'1000' => 'одной тысяче долларов',
				'2000' => 'двум тысячам долларов',

				'1000000' => 'одному миллиону долларов',
				'2000000' => 'двум миллионам долларов',

				'1000000000' => 'одному миллиарду долларов',
				'2000000000' => 'двум миллиардам долларов',

				'1000000000000' => 'одному триллиону долларов',
				'2000000000000' => 'двум триллионам долларов',

				'1002' => 'одной тысяче двум долларам',
				'1000000000005' => 'одному триллиону пяти долларам',

				'930651342187216' => 'девятистам тридцати триллионам шестистам пятидесяти одному миллиарду трёхстам сорока двум миллионам ста восьмидесяти семи тысячам двумстам шестнадцати долларам',

				'plural' => {	# множественное число
					'1' => 'одним долларам',
					'1000' => 'одним тысячам долларов',	# ам ?
					'1000000' => 'одним миллионам долларов',	# ам ?
					'1000000000' => 'одним миллиардам долларов',	# ам ?
					'1000000000000' => 'одним триллионам долларов',	# ам ?
				},

			},
			'a' => { # accusative (Винительный падеж): inanimate (неодушевлённый объект) вижу кого? что?
				'0'  => 'ноль долларов',	# 0..19 : неодушевлённый объект, как Именительный падеж
				'1'  => 'один доллар',
				'2'  => 'два доллара',
				'5'  => 'пять долларов',
				'10' => 'десять долларов',
				'11' => 'одиннадцать долларов',
				'100' => 'сто долларов',
				'1000' => 'одну тысячу долларов',
				'2000' => 'две тысячи долларов',

				'1000000' => 'один миллион долларов',
				'2000000' => 'два миллиона долларов',

				'1000000000' => 'один миллиард долларов',
				'2000000000' => 'два миллиарда долларов',

				'1000000000000' => 'один триллион долларов',
				'2000000000000' => 'два триллиона долларов',

				'1002' => 'одну тысячу два доллара',
				'1000000000005' => 'один триллион пять долларов',

				'930651342187216' => 'девятьсот тридцать триллионов шестьсот пятьдесят один миллиард триста сорок два миллиона сто восемьдесят семь тысяч двести шестнадцать долларов',

				'plural' => {	# множественное число
					'1'  => 'одни доллары',
					'1000' => 'одни тысячи долларов',
					'1000000' => 'одни миллионы долларов',
					'1000000000' => 'одни миллиарды долларов',
					'1000000000000' => 'одни триллионы долларов',
				},

			},
			'i' => { # instrumental (Творительный падеж) : оплачу кем? чем?
				'0'  => 'нолём долларов',
				'1'  => 'одним долларом',
				'2'  => 'двумя долларами',
				'5'  => 'пятью долларами',
				'10' => 'десятью долларами',
				'11' => 'одиннадцатью долларами',
				'100' => 'ста долларами',
				'1000' => 'одной тысячей долларов',
				'2000' => 'двумя тысячами долларов',

				'1000000' => 'одним миллионом долларов',
				'2000000' => 'двумя миллионами долларов',

				'1000000000' => 'одним миллиардом долларов',
				'2000000000' => 'двумя миллиардами долларов',

				'1000000000000' => 'одним триллионом долларов',
				'2000000000000' => 'двумя триллионами долларов',

				'1002' => 'одной тысячей двумя долларами',
				'1000000000005' => 'одним триллионом пятью долларами',

				'930651342187216' => 'девятьюстами тридцатью триллионами шестьюстами пятьюдесятью одним миллиардом тремястами сорока двумя миллионами ста восемьюдесятью семью тысячами двумястами шестнадцатью долларами',

				'plural' => {	# множественное число
					'1' => 'одними долларами',
					'1000' => 'одними тысячами долларов',
					'1000000' => 'одними миллионами долларов',
					'1000000000' => 'одними миллиардами долларов',
					'1000000000000' => 'одними триллионами долларов',
				},

			},
			'p' => { # prepositional (Предложный падеж) : думаю о ком? о чём?
				'0'  => 'ноле долларов',	# ах ?
				'1'  => 'одном долларе',
				'2'  => 'двух долларах',
				'5'  => 'пяти долларах',
				'10' => 'десяти долларах',
				'11' => 'одиннадцати долларах',
				'100' => 'ста долларах',

				'1000' => 'одной тысяче долларов',	# ах ?
				'2000' => 'двух тысячах долларов',	# ах ?

				'1000000' => 'одном миллионе долларов',	# ах ?
				'2000000' => 'двух миллионах долларов',	# ах ?

				'1000000000' => 'одном миллиарде долларов',	# ах ?
				'2000000000' => 'двух миллиардах долларов',	# ах ?

				'1000000000000' => 'одном триллионе долларов',	# ах ?
				'2000000000000' => 'двух триллионах долларов',	# ах ?

				'1002' => 'одной тысяче двух долларах',
				'1000000000005' => 'одном триллионе пяти долларах',

				'930651342187216' => 'девятистах тридцати триллионах шестистах пятидесяти одном миллиарде трёхстах сорока двух миллионах ста восьмидесяти семи тысячах двухстах шестнадцати долларах',

				'plural' => {	# множественное число
					'1' => 'одних долларах',
					'1000' => 'одних тысячах долларов',
					'1000000' => 'одних миллионах долларов',
					'1000000000' => 'одних миллиардах долларов',
					'1000000000000' => 'одних триллионах долларов',
				},
			},

		},
		'usd' => {
			'n' => {
				'1'  => 'один цент',
			},

		},
		'CNY' => {
			'n' => {
				'3'  => 'три юаня',
			},

		},
		'фынь' => {
			'n' => {
				'3'  => 'три фыня',
			},

		},
		'год' => {
			'n' => {
				'10'  => 'десять лет',
			},

		},
		'месяц' => {
			'n' => {
				'5'  => 'пять месяцев',
			},

		},
		'день' => {
			'n' => {
				'5'  => 'пять дней',
			},

		},
		'час' => {
			'n' => {
				'5'  => 'пять часов',
			},

		},
		'минута' => {
			'n' => {
				'3'  => 'три минуты',
			},

		},
		'секунда' => {
			'n' => {
				'3'  => 'три секунды',
			},

		},
		'метр' => {
			'n' => {
				'11'  => 'одиннадцать метров',
			},

		},
		'ПЕЧАТЬ' => {
			'n' => {
				'3'  => 'три печати',
			},

		},
		'000' => {	# Unknown!!!
			'n' => { # nominative (Именительный падеж): есть кто? что?
				'0'  => 'ноль',

				'plural' => {	# множественное число
					'1'  => 'одни',
				},
			},
		},

	}
}


sub datest_ucfirst {

	return {
		'n' => { # nominative (Именительный падеж): кто? что?
			'masculine' => { # Мужской род
				'0'  => 'Ноль',
				'1'  => 'Один',
				'2'  => 'Два',
				'3'  => 'Три',
				'4'  => 'Четыре',
				'5'  => 'Пять',
				'100' => 'Сто',
				'1000' => 'Одна тысяча',
				'1000000' => 'Один миллион',
			},
		}, #--------------------------------------------
		'g' => { # genitive (Родительный падеж): кого? чего?
			'prolog' => ['в'],
			'masculine' => { # Мужской род
				'1'  => 'в Одного',
				'2'  => 'в Двух',
				'3'  => 'в Трёх',
				'4'  => 'в Четырёх',
				'5'  => 'в Пяти',
				'100' => 'в Ста',
				'1000' => 'в Одной тысячи',
				'1000000' => 'в Одного миллиона',
			},
			'feminine' => { # Женский род
				'1'  => 'в Одной',
				'2'  => 'в Двух',	# like 'masculine'
			},
			'neuter' => { # Средний род
				'4'  => 'в Четырёх',	# ...
			},

			'plural' => {	# множественное число
				'1' => 'в Одних',
			},

		}, #--------------------------------------------
		'd' => { # dative (Дательный падеж): кому? чему?
			'epilog' => 'rub',
			'masculine' => { # Мужской род
				'0'  => 'Нолю копеек',
				'1'  => 'Одной копейке',
				'2'  => 'Двум копейкам',
				'5'  => 'Пяти копейкам',
				'10' => 'Десяти копейкам',
				'11' => 'Одиннадцати копейкам',
				'100' => 'Ста копейкам',
				'1000' => 'Одной тысяче копеек',
				'2000' => 'Двум тысячам копеек',
			},

		}, #--------------------------------------------
		'i' => { # instrumental (Творительный падеж) : кем? чем?
			'prolog' => 'с', # ['с','со=с'],
			'masculine' => { # Мужской род, default
				'0'  => 'с Нолём',
				'1'  => 'с Одним',
				'2'  => 'с Двумя',
				'3'  => 'с Тремя',
				'4'  => 'с Четырьмя',
				'5'  => 'с Пятью',

				'10' => 'с Десятью',
				'11' => 'с Одиннадцатью',
				'12' => 'с Двенадцатью',

				'20' => 'с Двадцатью',
				'100' => 'со Ста',

				'1000' => 'с Одной тысячей',
				'2000' => 'с Двумя тысячами',
				'100000' => 'со Ста тысячами',

				'1000000' => 'с Одним миллионом',
				'2000000' => 'с Двумя миллионами',
				'100000000' => 'со Ста миллионами',

				'1000000000' => 'с Одним миллиардом',
				'2000000000' => 'с Двумя миллиардами',
				'100000000000' => 'со Ста миллиардами',

				'1000000000000' => 'с Одним триллионом',
				'2000000000000' => 'с Двумя триллионами',
				'100000000000000' => 'со Ста триллионами',

				'930651342187216' => 'с Девятьюстами тридцатью триллионами шестьюстами пятьюдесятью одним миллиардом тремястами сорока двумя миллионами ста восемьюдесятью семью тысячами двумястами шестнадцатью',

			},
			'feminine' => { # Женский род: 'одной', а остальные как для Мужского рода
				'0'  => 'с Нолём',	# like 'masculine'
				'1'  => 'с Одной',	###
				'1000' => 'с Одной тысячей',
			},

			'plural' => {	# множественное число
				'1' => 'с Одними',
				'1000' => 'с Одними тысячами',
				'1000000' => 'с Одними миллионами',
				'1000000000' => 'с Одними миллиардами',
				'1000000000000' => 'с Одними триллионами',
			},

		}, #--------------------------------------------
		'p' => { # prepositional (Предложный падеж) : о ком? о чём?
			'prolog' => 'о', # ['о','об=од'],
			'masculine' => { # Мужской род
				'0'  => 'о Ноле',
				'1'  => 'об Одном',
				'2'  => 'о Двух',
				'3'  => 'о Трёх',
				'4'  => 'о Четырёх',
				'5'  => 'о Пяти',

				'10' => 'о Десяти',
				'11' => 'об Одиннадцати',
				'12' => 'о Двенадцати',

				'20' => 'о Двадцати',
				'100' => 'о Ста',

				'1000' => 'об Одной тысяче',
				'2000' => 'о Двух тысячах',
				'11000' => 'об Одиннадцати тысячах',
				'100000' => 'о Ста тысячах',

				'1000000' => 'об Одном миллионе',
				'2000000' => 'о Двух миллионах',
				'11000000' => 'об Одиннадцати миллионах',
				'100000000' => 'о Ста миллионах',

				'1000000000' => 'об Одном миллиарде',
				'2000000000' => 'о Двух миллиардах',

				'1000000000000' => 'об Одном триллионе',
				'2000000000000' => 'о Двух триллионах',

				'1002' => 'об Одной тысяче двух',
				'930651342187216' => 'о Девятистах тридцати триллионах шестистах пятидесяти одном миллиарде трёхстах сорока двух миллионах ста восьмидесяти семи тысячах двухстах шестнадцати',

			},
			'feminine' => { # Женский род: 'одной', а остальные как для Мужского рода
				'1'  => 'об Одной',	###
				'1000' => 'об Одной тысяче',
				'2000' => 'о Двух тысячах',
				'11000' => 'об Одиннадцати тысячах',
			},

			'plural' => {	# prepositional (Предложный падеж): множественное число
				'1' => 'об Одних', # неодушевлённый и одушевлённый
				'1000' => 'об Одних тысячах',
				'1000000' => 'об Одних миллионах',
			},

		},

	}

}

