# -*- coding: utf-8 -*-
# Moovida - Home multimedia server
# Copyright (C) 2006-2009 Fluendo Embedded S.L. (www.fluendo.com).
# All rights reserved.
#
# This file is available under one of two license agreements.
#
# This file is licensed under the GPL version 3.
# See "LICENSE.GPL" in the root of this distribution including a special
# exception to use Moovida with Fluendo's plugins.
#
# The GPL part of Moovida is also available under a commercial licensing
# agreement from Fluendo.
# See "LICENSE.Moovida" in the root directory of this distribution package
# for details on that license.
#
# Authors: Florian Boucault <florian@fluendo.com>
#          Olivier Tilloy <olivier@fluendo.com>

from twisted.trial.unittest import TestCase

from elisa.core import common
from elisa.core.utils import defer

from elisa.plugins.poblesec.browser import BrowserController, EmptyHistory, \
    TRANSITION_FORWARD_IN, TRANSITION_FORWARD_OUT, \
    TRANSITION_BACKWARD_IN, TRANSITION_BACKWARD_OUT
from elisa.plugins.pigment.pigment_frontend import ControllerNotFound
from elisa.plugins.pigment.pigment_controller import PigmentController

from twisted.internet import task

class StubConfig(object):
    def get_section(self, section_name, default=None):
        return {}


class StubApplication(object):
    """
    Used to provide logging facilities.
    """
    def __init__(self):
        self.config = StubConfig()
        self.log_failure_called = False

    def log_failure(self, failure):
        self.log_failure_called = True


# Fake controllers
class SimpleController(PigmentController):
    pass


class MockArgsController(PigmentController):
    def initialize(self, **kwargs):
        self.kwargs = kwargs
        return super(MockArgsController, self).initialize()


class BadControllerError(Exception):
    pass


class BadController(PigmentController):
    def __init__(self):
        super(BadController, self).__init__()
        raise BadControllerError()


class NeverEndingController(PigmentController):
    def initialize(self):
        return defer.Deferred()


class StubController(PigmentController):

    def __init__(self):
        super(StubController, self).__init__()
        self.prepare_called = False
        self.removed_called = False
        self.clean_called = False

    def clean(self):
        self.clean_called = True

    def prepare(self):
        self.prepare_called = True

    def removed(self):
        self.removed_called = True


class FakePigmentFrontend(object):
    """
    Used to provide controller's creation facilities.
    """

    controllers = {'/simple/controller': SimpleController,
                   '/bad/controller': BadController,
                   '/never/ending/controller': NeverEndingController,
                   '/mock/args/controller': MockArgsController,
                   '/stub/controller': StubController,
                  }

    def create_controller(self, path, **kwargs):
        try:
            klass = self.controllers[path]
        except KeyError:
            raise ControllerNotFound(path)

        return klass.create(**kwargs)


class StubTransition(object):
    path = 'elisa.plugins.poblesec.tests.test_browser.StubTransition'

    def __init__(self):
        self.apply_called = False

    def apply(self, controller):
        self.apply_called = True
        return defer.succeed(controller)


class TestBrowserController(TestCase):

    test_config = {'home': '',
                   TRANSITION_FORWARD_IN: StubTransition.path,
                   TRANSITION_FORWARD_OUT: StubTransition.path,
                   TRANSITION_BACKWARD_IN: StubTransition.path,
                   TRANSITION_BACKWARD_OUT: StubTransition.path
                  }

    def setUp(self):
        self._patch_application()
        self.browser = BrowserController()
        self.browser.config = self.test_config
        self.browser.frontend = FakePigmentFrontend()
        return self.browser.initialize()

    def tearDown(self):
        dfr = self.browser.clean()
        dfr.addBoth(lambda result: self._unpatch_application())
        return dfr

    def _patch_application(self):
        self.__application = common.application
        common.application = StubApplication()

    def _unpatch_application(self):
        common.application = self.__application
        del self.__application


class TestBrowserControllerLoadNewController(TestBrowserController):

    def regular_checks(self, path, result):
        klass = FakePigmentFrontend.controllers[path]
        browser_klass = self.browser.__class__
        self.failUnless(isinstance(result, klass))
        self.failUnlessIdentical(result, self.browser.current_controller)
        self.failUnless(isinstance(result.browser, browser_klass))

    def test_empty_path(self):
        path = ''
        dfr = self.browser.load_new_controller(path)
        self.failUnless(common.application.log_failure_called)
        return self.failUnlessFailure(dfr, ControllerNotFound)

    def test_invalid_path(self):
        path = 'this-is.noTa/Path'
        dfr = self.browser.load_new_controller(path)
        self.failUnless(common.application.log_failure_called)
        return self.failUnlessFailure(dfr, ControllerNotFound)

    def test_non_existent_path(self):
        path = '/this/path/does/not/exist'
        dfr = self.browser.load_new_controller(path)
        self.failUnless(common.application.log_failure_called)
        return self.failUnlessFailure(dfr, ControllerNotFound)

    def test_simple(self):
        path = '/simple/controller'
        dfr = self.browser.load_new_controller(path)

        def checks(result):
            self.regular_checks(path, result)

        dfr.addCallback(checks)
        return dfr

    def test_args_passing(self):
        path = '/mock/args/controller'
        dfr = self.browser.load_new_controller(path, arg1='test1', arg2='test2')

        def checks(result):
            self.regular_checks(path, result)
            self.failUnlessIdentical(result.kwargs['arg1'], 'test1')
            self.failUnlessIdentical(result.kwargs['arg2'], 'test2')

        dfr.addCallback(checks)
        return dfr

    def test_with_hints(self):
        path = '/mock/args/controller'
        hints = ['hint1', 'hint2']
        dfr = self.browser.load_new_controller(path, hints=hints)

        def checks(result):
            self.regular_checks(path, result)
            self.failUnlessEquals(len(result.kwargs), 0)

        dfr.addCallback(checks)
        return dfr

    def test_with_hints_and_args(self):
        path = '/mock/args/controller'
        hints = ['hint1', 'hint2']
        dfr = self.browser.load_new_controller(path, hints=hints, arg1='test1',
                                               arg2='test2')

        def checks(result):
            self.regular_checks(path, result)
            self.failUnlessEquals(len(result.kwargs), 2)
            self.failUnlessIdentical(result.kwargs['arg1'], 'test1')
            self.failUnlessIdentical(result.kwargs['arg2'], 'test2')

        dfr.addCallback(checks)
        return dfr

    def test_bad_controller(self):
        path = '/bad/controller'
        dfr = self.browser.load_new_controller(path)
        return self.failUnlessFailure(dfr, BadControllerError)

    def test_multiple_controllers(self):
        """
        Test loading a series of 2 valid controllers
        """
        path = '/simple/controller'
        dfr = self.browser.load_new_controller(path)

        def checks_simple_controller(result):
            self.regular_checks(path, result)

        def create_simple_controller(result):
            return self.browser.load_new_controller(path)

        dfr.addCallback(checks_simple_controller)
        dfr.addCallback(create_simple_controller)
        dfr.addCallback(checks_simple_controller)

        return dfr

    def test_current_controller_on_error(self):
        """
        Test the current_controller after the following sequence:
        - loading of a valid controller
        - failed loading of a bad controller
        """
        path = '/simple/controller'
        dfr = self.browser.load_new_controller(path)

        def checks_simple_controller(result):
            self.regular_checks(path, result)
            # save the controller for later checks
            self.created_controller = result

        def create_bad_controller(result):
            path = '/this/path/does/not/exist'
            dfr = self.browser.load_new_controller(path)
            return dfr

        def checks_bad_controller(result):
            # ensure that the current controller has not changed
            current = self.browser.current_controller
            self.failUnlessIdentical(self.created_controller, current)

        dfr.addCallback(checks_simple_controller)
        dfr.addCallback(create_bad_controller)
        dfr.addErrback(checks_bad_controller)
        return dfr

    def test_cancellation(self):
        """
        Test that cancellation works with the following sequence:
        - loading of a valid controller
        - loading of another valid controller
        """
        path = '/simple/controller'
        dfr_first = self.browser.load_new_controller(path)

        def checks(result):
            self.regular_checks(path, result)
            self.failUnless(dfr_first.called)

        path = '/simple/controller'
        dfr_second = self.browser.load_new_controller(path)

        dfr_second.addCallback(checks)

        return defer.DeferredList([dfr_second, dfr_first])

    def test_cancellation_bad_controller(self):
        """
        Test the current_controller after the following sequence:
        - failed loading of a bad controller
        - loading of a valid controller
        """
        path = '/bad/controller'
        dfr_bad = self.browser.load_new_controller(path)
        dfr_bad = self.failUnlessFailure(dfr_bad, BadControllerError)

        def checks(result):
            self.regular_checks(path, result)
            self.failUnless(dfr_bad.called)

        path = '/simple/controller'
        dfr_simple = self.browser.load_new_controller(path)

        dfr_simple.addCallback(checks)

        return defer.DeferredList([dfr_simple, dfr_bad])

    def test_cancellation_never_ending_controller(self):
        """
        Test that cancellation works with the following sequence:
        - loading a controller that never finishes initialising
        - loading of a valid controller
        """
        path = '/never/ending/controller'
        dfr_never_ending = self.browser.load_new_controller(path)

        def checks(result):
            self.regular_checks(path, result)
            self.failUnless(dfr_never_ending.called)

        path = '/simple/controller'
        dfr_simple = self.browser.load_new_controller(path)

        dfr_simple.addCallback(checks)

        return defer.DeferredList([dfr_simple, dfr_never_ending])

    test_cancellation_never_ending_controller.timeout = 2

    def test_transitions_creation(self):
        self.failUnless(isinstance(self.browser._transition_forward_in,
                                   StubTransition))
        self.failUnless(isinstance(self.browser._transition_forward_out,
                                   StubTransition))
        self.failUnless(isinstance(self.browser._transition_backward_in,
                                   StubTransition))
        self.failUnless(isinstance(self.browser._transition_backward_out,
                                   StubTransition))


class TestBrowserControllerTransitions(TestBrowserController):

    def test_show_controller(self):
        controller = StubController()
        transition = StubTransition()

        dfr = self.browser._show_controller(controller, transition)

        self.failUnless(transition.apply_called)
        self.failUnless(controller.prepare_called)
        self.failUnless(controller.widget.focus)
        self.failUnless(controller.sensitive)
        self.failUnlessIn(controller.widget, self.browser.widget)
        return dfr

    def test_hide_controller(self):
        controller = StubController()
        transition = StubTransition()

        # simulate showing controller
        self.browser.widget.add(controller.widget)
        controller.widget.focus = True
        controller.sensitive = True

        dfr = self.browser._hide_controller(controller, transition)

        self.failUnless(transition.apply_called)
        self.failUnless(controller.removed_called)
        self.failIf(controller.sensitive)
        self.failIfIn(controller.widget, self.browser.widget)
        return dfr


class TestBrowserControllerLoadPreviousController(TestBrowserController):

    def setUp(self):
        dfr = super(TestBrowserControllerLoadPreviousController, self).setUp()

        def done(result):
            self.show_controller_called = {}
            self._patch_browser()

        dfr.addCallback(done)        
        return dfr

    def tearDown(self):
        dfr = super(TestBrowserControllerLoadPreviousController, self).tearDown()
        dfr.addCallback(lambda result: self._unpatch_browser())
        return dfr

    def _patch_browser(self):
        self.__browser_show_controller = self.browser._show_controller
        self.browser._show_controller = self._show_controller_stub
        self.__browser_hide_controller = self.browser._hide_controller
        self.browser._hide_controller = self._hide_controller_stub

    def _unpatch_browser(self):
        self.browser._show_controller = self.__browser_show_controller
        del self.__browser_show_controller
        self.browser._hide_controller = self.__browser_hide_controller
        del self.__browser_hide_controller

    def _show_controller_stub(self, controller, transition):
        controller.show_controller_called = True
        return defer.succeed(controller)

    def _hide_controller_stub(self, controller, transition):
        controller.hide_controller_called = True
        return defer.succeed(controller)

    def check_show_called(self, controller):
        self.failUnless(controller.show_controller_called)
        controller.show_controller_called = False

    def check_hide_called(self, controller):
        self.failUnless(controller.hide_controller_called)
        controller.hide_controller_called = False

    def test_load_previous_empty(self):
        """
        If no controller was loaded raise EmptyHistory
        """
        self.failUnlessRaises(EmptyHistory,
                              self.browser.load_previous_controller)

    def test_load_previous_one(self):
        """
        If only one controller was loaded raise EmptyHistory
        """
        dfr = self.browser.load_new_controller('/simple/controller')

        def load_previous(controller):
            self.failUnlessRaises(EmptyHistory,
                                  self.browser.load_previous_controller)

        dfr.addCallback(load_previous)
        return dfr

    def test_load_previous_simple(self):
        """
        Test chaining 2 loading of SimpleController and 1 load_previous
        """
        dfr = self.browser.load_new_controller('/stub/controller')

        def first_controller_created(controller):
            self.first_controller = controller
            self.check_show_called(self.first_controller)
            return self.browser.load_new_controller('/stub/controller')

        def second_controller_created(controller):
            self.second_controller = controller
            self.check_hide_called(self.first_controller)
            self.check_show_called(self.second_controller)
            return self.browser.load_previous_controller()

        def previous_loaded(result):
            self.check_hide_called(self.second_controller)
            self.check_show_called(self.first_controller)
            self.failUnless(self.second_controller.clean_called)

        def cleanup(result):
            del self.first_controller
            del self.second_controller

        dfr.addCallback(first_controller_created)
        dfr.addCallback(second_controller_created)
        dfr.addCallback(previous_loaded)
        dfr.addCallbacks(cleanup)
        return dfr


class TestBrowserControllerLoadInitialController(TestBrowserController):

    def _check_history(self, lenght):
        self.failUnlessEquals(len(self.browser._history), lenght)

    def _check_load_initial_controller(self, result):
        self._check_history(lenght=1)
        self.failUnlessIdentical(self.browser.current_controller.kwargs['arg'], 'first')
        self.failUnless(isinstance(self.browser.current_controller, MockArgsController))
        self.failUnlessRaises(EmptyHistory, self.browser.load_previous_controller)

    def test_done(self):
        def _load_controllers():
            def iterate_controllers():
                iteration = 0
                while iteration < 4:
                    iteration += 1
                    yield self.browser.load_new_controller('/simple/controller')

            dfr = self.browser.load_new_controller('/mock/args/controller',
                                                   arg='first')
            dfr.addCallback(lambda result: task.coiterate(iterate_controllers()))
            return dfr

        def controllers_loaded(result):
            self._check_history(lenght=5)
            dfr = self.browser.load_initial_controller()
            dfr.addCallback(self._check_load_initial_controller)
            return dfr

        dfr = _load_controllers()
        dfr.addCallback(controllers_loaded)
        return dfr
