/*
 * indicator-network - user interface for connman
 * Copyright 2010 Canonical Ltd.
 *
 * Authors:
 * Kalle Valo <kalle.valo@canonical.com>
 *
 * This program is free software: you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 3, as published
 * by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful, but
 * WITHOUT ANY WARRANTY; without even the implied warranties of
 * MERCHANTABILITY, SATISFACTORY QUALITY, or FITNESS FOR A PARTICULAR
 * PURPOSE.  See the GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License along
 * with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#include "ofono-manager.h"

#include <string.h>

#include "ofono.h"

G_DEFINE_TYPE(OfonoManager, ofono_manager, G_TYPE_OBJECT)

#define GET_PRIVATE(o) \
  (G_TYPE_INSTANCE_GET_PRIVATE((o), TYPE_OFONO_MANAGER, OfonoManagerPrivate))

typedef struct _OfonoManagerPrivate OfonoManagerPrivate;

struct _OfonoManagerPrivate {
  GDBusProxy *manager;
  UIProxy *ui;
  GHashTable *modems;
  GHashTable *sims;

  /* current_sim is also in priv->sims, so no need to unref this */
  GDBusProxy *current_sim;
};

#define PIN_TYPE_PIN "pin"

static void enter_pin_cb(GObject *object, GAsyncResult *res,
			 gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);
  OfonoManagerPrivate *priv = GET_PRIVATE(self);
  GDBusProxy *proxy = G_DBUS_PROXY(object);
  GError *error = NULL;
  GVariant *result;

  result = g_dbus_proxy_call_finish(proxy, res, &error);

  if (error != NULL) {
    g_warning("Ask pin failed: %s", error->message);
    g_error_free(error);
    goto out;
  }

  if (result == NULL) {
    g_warning("Ask pin failed, but no errors");
    goto out;
  }

  /* FIXME: if PIN fails, we should ask again */

  g_variant_unref(result);

 out:
  priv->current_sim = NULL;

  /* trick to avoid destroying self during async call */
  g_object_unref(self);

  /* FIXME: check if there are any pending sims needing pin */
}

static void enter_pin(OfonoManager *self, GDBusProxy *proxy,
		      const gchar *type, const gchar *pin)
{
  GVariant *parameters;

  parameters = g_variant_new("(ss)", type, pin);

  g_dbus_proxy_call(proxy, "EnterPin", parameters,
  		    G_DBUS_CALL_FLAGS_NONE, -1, NULL,
  		    enter_pin_cb, g_object_ref(self));
}

static void ask_pin_cb(GObject *object, GAsyncResult *res,
		       gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);
  OfonoManagerPrivate *priv = GET_PRIVATE(self);
  UIProxy *ui = UI_PROXY(object);
  GError *error = NULL;
  gchar *pin;

  g_return_if_fail(priv != NULL);

  pin = ui_proxy_ask_pin_finish(ui, res, &error);

  if (error != NULL) {
    g_warning("ask pin failed: %s\n", error->message);
    g_error_free(error);
    goto out;
  }

  g_return_if_fail(priv->current_sim != NULL);

  if (pin == NULL || strlen(pin) == 0) {
    g_warning("received empty pin");
    priv->current_sim = NULL;
    goto out;
  }

  enter_pin(self, priv->current_sim, PIN_TYPE_PIN, pin);

 out:
  g_free(pin);

  /* trick to avoid destroying self during async call */
  g_object_unref(self);
}

static void ask_pin(OfonoManager *self, GDBusProxy *proxy)
{
  OfonoManagerPrivate *priv = GET_PRIVATE(self);

  if (priv->current_sim != NULL) {
    /* there's currently another pin query ongoing, let's cancel this one */
    return;
  }

  priv->current_sim = proxy;
  ui_proxy_ask_pin(priv->ui, PIN_TYPE_PIN, NULL, ask_pin_cb,
		   g_object_ref(self));
}

static void update_pin_required(OfonoManager *self, GDBusProxy *proxy,
				GVariant *parameters)
{
  const gchar *state;

  state = g_variant_get_string(parameters, NULL);

  g_debug("%s(): state %s", __func__, state);

  if (g_strcmp0(state, PIN_TYPE_PIN) == 0) {
    /* sim requires pin now, ask from ui */
    ask_pin(self, proxy);
  }

  g_variant_unref(parameters);
}

static void update_sim_property(OfonoManager *self, GDBusProxy *proxy,
				const gchar *property, GVariant *value)
{
  if (g_strcmp0(property, OFONO_PROPERTY_PIN_REQUIRED) == 0)
    update_pin_required(self, proxy, value);
  else
    g_variant_unref(value);
}

/* callee owns parameters variant */
static void sim_property_changed(OfonoManager *self, GDBusProxy *proxy,
				 GVariant *parameters)
{
  GVariant *value;
  gchar *name;

  g_variant_get(parameters, "(sv)", &name, &value);

  /* callee takes the ownership of the value */
  update_sim_property(self, proxy, name, value);

  g_free(name);
}

static void sim_get_properties_cb(GObject *object, GAsyncResult *res,
				  gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);
  GDBusProxy *proxy = G_DBUS_PROXY(object);
  GVariant *result, *value, *dict;
  GVariantIter iter, dict_iter;
  GError *error = NULL;
  gchar *name;

  result = g_dbus_proxy_call_finish(proxy, res, &error);

  if (error != NULL) {
    g_warning("Failed to get sim properties from %s: %s",
	      g_dbus_proxy_get_object_path(proxy), error->message);
    g_error_free(error);
    goto out;
  }

  if (result == NULL) {
    g_warning("Failed to get sim properties from %s, but no errors",
	      g_dbus_proxy_get_object_path(proxy));
    goto out;
  }

  g_variant_iter_init(&iter, result);

  /* remove the tuple */
  g_variant_iter_init(&iter, result);
  dict = g_variant_iter_next_value(&iter);

  g_variant_iter_init(&dict_iter, dict);

  while (g_variant_iter_next(&dict_iter, "{sv}", &name, &value)) {
    /* callee owns the variant now */
    update_sim_property(self, proxy, name, value);

    g_free(name);
  }

  g_variant_unref(result);

 out:
  /* trick to avoid destroying self during async call */
  g_object_unref(self);
}

static void sim_g_signal(GDBusProxy *proxy, gchar *sender_name,
			   gchar *signal_name, GVariant *parameters,
			   gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);

  g_variant_ref(parameters);

  if (g_strcmp0(signal_name, OFONO_SIGNAL_PROPERTY_CHANGED) == 0)
    sim_property_changed(self, proxy, parameters);

  g_variant_unref(parameters);
}

static void create_sim_cb(GObject *object, GAsyncResult *res,
			  gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);
  OfonoManagerPrivate *priv = GET_PRIVATE(self);
  GError *error = NULL;
  GDBusProxy *proxy;
  const gchar *path;

  proxy = g_dbus_proxy_new_finish(res, &error);

  if (error != NULL) {
    g_warning("Failed to get sim proxy: %s", error->message);
    g_error_free(error);
    goto out;
  }

  if (proxy == NULL) {
    g_warning("Failed to get sim proxy, but no errors");
    goto out;
  }

  path = g_dbus_proxy_get_object_path(proxy);

  if (g_hash_table_remove(priv->sims, path))
    g_warning("%s(): sim '%s' already exists, removed it", __func__, path);

  g_debug("adding sim '%s'", path);
  g_hash_table_insert(priv->sims, g_strdup(path), proxy);

  g_signal_connect(proxy, "g_signal", G_CALLBACK(sim_g_signal),
		   self);

  g_dbus_proxy_call(proxy, "GetProperties", NULL,
  		    G_DBUS_CALL_FLAGS_NONE, -1, NULL,
  		    sim_get_properties_cb, g_object_ref(self));

 out:
  /* trick to avoid destroying self during async call */
  g_object_unref(self);
}

static void add_sim(OfonoManager *self, const gchar *path)
{
  g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
			   G_DBUS_PROXY_FLAGS_NONE,
			   NULL,
			   OFONO_SERVICE,
			   path,
			   OFONO_SIM_INTERFACE,
			   NULL,
			   create_sim_cb,
			   g_object_ref(self));
}

static void remove_sim(OfonoManager *self, const gchar *path)
{
  OfonoManagerPrivate *priv = GET_PRIVATE(self);
  GDBusProxy *proxy;

  g_debug("removing sim '%s'", path);

  proxy = g_hash_table_lookup(priv->sims, path);
  if (proxy == NULL)
    return;

  if (priv->current_sim == proxy)
    /* FIXME: cancel pending ui dbus calls */
    priv->current_sim = NULL;

  /* hash table frees the path and the modem */
  g_hash_table_remove(priv->sims, path);
}

static void update_modem_interfaces(OfonoManager *self, GDBusProxy *proxy,
				    GVariant *variant)
{
  gboolean found;
  GVariantIter iter;
  const gchar *path;
  gchar *interface;

  path = g_dbus_proxy_get_object_path(proxy);

  g_variant_iter_init(&iter, variant);
  found = FALSE;

  while (g_variant_iter_next(&iter, "s", &interface)) {
    if (g_strcmp0(interface, OFONO_SIM_INTERFACE) == 0)
      found = TRUE;

    g_free(interface);
  }

  if (found)
    add_sim(self, path);
  else
    /* sim interface can be removed anytime */
    remove_sim(self, path);
}

static void modem_property_changed(OfonoManager *self, GDBusProxy *proxy,
				   GVariant *parameters)
{
  GVariant *value;
  gchar *name;

  g_variant_get(parameters, "(sv)", &name, &value);

  if (g_strcmp0(name, OFONO_PROPERTY_INTERFACES) == 0)
    update_modem_interfaces(self, proxy, value);

  g_variant_unref(value);
  g_free(name);
}

static void modem_get_properties_cb(GObject *object, GAsyncResult *res,
				    gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);
  GDBusProxy *proxy = G_DBUS_PROXY(object);
  GVariant *result, *value, *dict;
  GVariantIter iter, dict_iter;
  GError *error = NULL;
  gchar *key;

  result = g_dbus_proxy_call_finish(proxy, res, &error);

  if (error != NULL) {
    g_warning("Failed to get modem properties from %s: %s",
	      g_dbus_proxy_get_object_path(proxy), error->message);
    g_error_free(error);
    goto out;
  }

  if (result == NULL) {
    g_warning("Failed to get modem properties from %s, but no errors",
	      g_dbus_proxy_get_object_path(proxy));
    goto out;
  }

  /* result is (a{sv}) */
  g_variant_iter_init(&iter, result);

  /* remove the tuple */
  g_variant_iter_init(&iter, result);
  dict = g_variant_iter_next_value(&iter);

  g_variant_iter_init(&dict_iter, dict);

  /* find security type */
  while (g_variant_iter_next(&dict_iter, "{sv}", &key, &value)) {
    if (g_strcmp0(key, OFONO_PROPERTY_INTERFACES) == 0)
      update_modem_interfaces(self, proxy, value);

    g_variant_unref(value);
    g_free(key);
  }

  g_variant_unref(result);

 out:
  /* trick to avoid destroying self during async call */
  g_object_unref(self);
}

static void modem_g_signal(GDBusProxy *proxy, gchar *sender_name,
			   gchar *signal_name, GVariant *parameters,
			   gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);

  /*
   * gdbus documentation is not clear who owns the variant so take it, just
   * in case
   */
  g_variant_ref(parameters);

  if (g_strcmp0(signal_name, OFONO_SIGNAL_PROPERTY_CHANGED) == 0)
    modem_property_changed(self, proxy, parameters);

  g_variant_unref(parameters);
}

static void create_modem_cb(GObject *object, GAsyncResult *res,
			    gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);
  OfonoManagerPrivate *priv = GET_PRIVATE(self);
  GError *error = NULL;
  const gchar *path;
  GDBusProxy *proxy;

  proxy = g_dbus_proxy_new_finish(res, &error);

  if (error != NULL) {
    g_warning("Failed to get modem proxy: %s", error->message);
    g_error_free(error);
    goto out;
  }

  if (proxy == NULL) {
    g_warning("Failed to get modem proxy, but no errors");
    goto out;
  }

  path = g_dbus_proxy_get_object_path(proxy);

  if (g_hash_table_remove(priv->modems, path))
    g_warning("%s(): modem '%s' exists already, removed",__func__, path);

  g_debug("adding modem '%s'", path);
  g_hash_table_insert(priv->modems, g_strdup(path), proxy);

  g_signal_connect(proxy, "g_signal", G_CALLBACK(modem_g_signal),
		   self);

  g_dbus_proxy_call(proxy, "GetProperties", NULL,
  		    G_DBUS_CALL_FLAGS_NONE, -1, NULL,
  		    modem_get_properties_cb, g_object_ref(self));

 out:
  /* trick to avoid destroying self during async call */
  g_object_unref(self);
}

static void add_modem(OfonoManager *self, const gchar *path)
{
  g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM,
			   G_DBUS_PROXY_FLAGS_NONE,
			   NULL,
			   OFONO_SERVICE,
			   path,
			   OFONO_MODEM_INTERFACE,
			   NULL,
			   create_modem_cb,
			   g_object_ref(self));
}

static void remove_modem(OfonoManager *self, const gchar *path)
{
  OfonoManagerPrivate *priv = GET_PRIVATE(self);

  g_debug("removing modem '%s'", path);

  /* hash table frees the path and the modem */
  g_hash_table_remove(priv->modems, path);
}

static void manager_modem_added(OfonoManager *self, GVariant *parameters)
{
  GVariantIter iter;
  gchar *path;

  g_variant_iter_init(&iter, parameters);
  g_variant_iter_next(&iter, "o", &path);

  add_modem(self, path);

  g_free(path);
}

static void manager_modem_removed(OfonoManager *self, GVariant *parameters)
{
  GVariantIter iter;
  gchar *path;

  g_variant_iter_init(&iter, parameters);
  g_variant_iter_next(&iter, "o", &path);

  remove_sim(self, path);
  remove_modem(self, path);

  g_free(path);
}

static void manager_g_signal(GDBusProxy *proxy, gchar *sender_name,
			     gchar *signal_name, GVariant *parameters,
			     gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);

  /*
   * gdbus documentation is not clear who owns the variant so take it, just
   * in case
   */
  g_variant_ref(parameters);

  if (g_strcmp0(signal_name, OFONO_SIGNAL_MODEM_ADDED) == 0)
    manager_modem_added(self, parameters);
  else if (g_strcmp0(signal_name, OFONO_SIGNAL_MODEM_REMOVED) == 0)
    manager_modem_removed(self, parameters);

  g_variant_unref(parameters);
}

static void manager_get_modems_cb(GObject *object, GAsyncResult *res,
				  gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);
  GDBusProxy *proxy = G_DBUS_PROXY(object);
  GVariantIter iter, array_iter, tuple_iter;
  GVariant *result, *array, *array_value, *tuple_value;
  GError *error = NULL;
  gchar *path;

  result = g_dbus_proxy_call_finish(proxy, res, &error);

  if (error != NULL) {
    g_warning("Failed to get ofono modems: %s", error->message);
    g_error_free(error);
    goto out;
  }

  if (result == NULL) {
    g_warning("Failed to get ofono modems but no error");
    goto out;
  }

  /* result will be "(a(oa{sv}))" */

  /* remove the tuple */
  g_variant_iter_init(&iter, result);
  array = g_variant_iter_next_value(&iter);

  g_variant_iter_init(&array_iter, array);

  while ((array_value = g_variant_iter_next_value(&array_iter)) != NULL) {
    g_variant_iter_init(&tuple_iter, array_value);
    tuple_value = g_variant_iter_next_value(&tuple_iter);
    g_variant_get(tuple_value, "o", &path);

    add_modem(self, path);
    g_free(path);
  }

  g_variant_unref(result);

 out:
  /* trick to avoid destroying self during async call */
  g_object_unref(self);
}

static void create_manager_cb(GObject *object, GAsyncResult *res,
			      gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);
  OfonoManagerPrivate *priv = GET_PRIVATE(self);
  GError *error = NULL;

  priv->manager = g_dbus_proxy_new_finish(res, &error);

  if (error != NULL) {
    g_warning("Failed to get ofono manager proxy: %s", error->message);
    g_error_free(error);
    goto out;
  }

  if (priv->manager == NULL) {
    g_warning("Failed to get ofono manager proxy, but no errors");
    goto out;
  }

  g_signal_connect(priv->manager, "g_signal", G_CALLBACK(manager_g_signal),
		   self);

  g_dbus_proxy_call(priv->manager, "GetModems", NULL,
  		    G_DBUS_CALL_FLAGS_NONE, -1, NULL,
  		    manager_get_modems_cb, g_object_ref(self));

 out:
  /* trick to avoid destroying self during async call */
  g_object_unref(self);
}

static void create_manager(OfonoManager *self)
{
  OfonoManagerPrivate *priv = GET_PRIVATE(self);

  if (priv->manager != NULL)
    return;

  /* FIXME: handle ofono reboots and crashes */

  g_dbus_proxy_new_for_bus(G_BUS_TYPE_SYSTEM, G_DBUS_PROXY_FLAGS_NONE,
			   NULL,
			   OFONO_SERVICE,
			   OFONO_MANAGER_PATH,
			   OFONO_MANAGER_INTERFACE,
			   NULL,
			   create_manager_cb, g_object_ref(self));
}

static void destroy_proxies(OfonoManager *self)
{
  OfonoManagerPrivate *priv = GET_PRIVATE(self);

  if (priv->manager != NULL) {
    g_object_unref(priv->manager);
    priv->manager = NULL;
  }

  if (priv->modems != NULL) {
    g_hash_table_destroy(priv->modems);
    priv->modems = NULL;
  }

  if (priv->sims != NULL) {
    g_hash_table_destroy(priv->sims);
    priv->sims = NULL;
  }
}

static void ui_connected_notify(UIProxy *ui, GParamSpec *pspec,
				gpointer user_data)
{
  OfonoManager *self = OFONO_MANAGER(user_data);

  if (ui_proxy_is_connected(ui))
    create_manager(self);
  else
    destroy_proxies(self);
}

static void ofono_manager_dispose(GObject *object)
{
  OfonoManager *self = OFONO_MANAGER(object);
  OfonoManagerPrivate *priv = GET_PRIVATE(self);

  destroy_proxies(self);

  if (priv->ui != NULL) {
    g_object_unref(priv->ui);
    priv->ui = NULL;
  }

  G_OBJECT_CLASS(ofono_manager_parent_class)->dispose(object);
}

static void ofono_manager_finalize(GObject *object)
{
  G_OBJECT_CLASS(ofono_manager_parent_class)->finalize(object);
}

static void ofono_manager_class_init(OfonoManagerClass *klass)
{
  GObjectClass *object_class = G_OBJECT_CLASS(klass);

  g_type_class_add_private(klass, sizeof(OfonoManagerPrivate));

  object_class->dispose = ofono_manager_dispose;
  object_class->finalize = ofono_manager_finalize;
}

static void ofono_manager_init(OfonoManager *self)
{
  OfonoManagerPrivate *priv = GET_PRIVATE(self);

  priv->manager = NULL;
  priv->current_sim = NULL;

  priv->modems = g_hash_table_new_full(g_str_hash, g_str_equal,
				       g_free, g_object_unref);
  priv->sims = g_hash_table_new_full(g_str_hash, g_str_equal,
				     g_free, g_object_unref);
}

OfonoManager *ofono_manager_new(Manager *manager)
{
  OfonoManager *self;
  OfonoManagerPrivate *priv;

  g_debug("%s", __func__);

  self = g_object_new(TYPE_OFONO_MANAGER, NULL);

  priv = GET_PRIVATE(self);
  priv->ui = manager_get_ui(manager);
  g_object_ref(priv->ui);

  g_signal_connect(priv->ui, "notify::connected",
		   G_CALLBACK(ui_connected_notify),
		   self);

  g_return_val_if_fail(priv->ui != NULL, NULL);

  if (ui_proxy_is_connected(priv->ui))
    create_manager(self);

  return self;
}
