/* socket.hh - network sockets for use with shevek::main
 * Copyright 2003 Bas Wijnen <wijnen@debian.org>
 *
 * This program is free software: you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation, either version 3 of the License, or
 * (at your option) any later version.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program.  If not, see <http://www.gnu.org/licenses/>.
 */

#ifndef SHEVEK_SOCKET_HH
#define SHEVEK_SOCKET_HH

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/un.h>
#include <netdb.h>
#include <sigc++/sigc++.h>
#include <glibmm.h>
#include "debug.hh"
#include "fd.hh"
#include "error.hh"
#include "avahi.hh"

namespace shevek
{
  /// Use a unix-domain, tcp or avahi network connection with shevek::fd.
  class socket : public fd
  {
  public:
    /// Disconnect signal type.
    typedef sigc::signal0 <void> disconnect_t;
    /// New connection callback type.
    typedef sigc::slot0 <void> listen_t;
    /// Create a new socket.
    static Glib::RefPtr <socket> create (Glib::RefPtr <Glib::MainContext> main = Glib::MainContext::get_default () );
    /// Listen for new connections on a UNIX socket.  Use listen instead.
    void listen_unix (std::string const &file, listen_t cb, unsigned queue = 10);
    /// Listen for new connections on a TCP socket.  Use listen instead.
    void listen_tcp (std::string const &service, listen_t cb, unsigned queue = 10);
    /// Listen for new connections on a TCP socket, and register it with avahi.  Use listen instead.
    void listen_avahi (std::string const &service, Glib::ustring const &protocol, Glib::ustring const &name, listen_t cb, unsigned queue = 10);
    /// Listen for new connections.
    /** This is the preferred function to use.
     *  Format:
     *  UNIX domain sockets: anything with at least one / in it.
     *  TCP services: the name.
     *  TCP port numbers: the number.
     *  For TCP, appending |name|protocol, where name is the application name and protocol the connection type.
     */
    void listen (std::string const &port, listen_t cb, unsigned queue = 10);
    /// Connect to a UNIX socket.  Use connect instead.
    void connect_unix (std::string const &unix_name);
    /// Connect to a TCP socket.  Use connect instead.
    void connect_tcp (std::string const &host, std::string const &service);
    /// Connect to an avahi TCP socket.  Use connect instead.
    void connect_avahi (avahi::browser::owner const &target, avahi::browser::details const &details = avahi::browser::details ());
    /// Connect to a socket.
    /** This is the preferred function to use.
     *  Format:
     *  UNIX domain sockets: anything with at least one / in it.
     *  TCP: hostname:port, where the hostname and colon may be omitted, and the port may be a service or number.
     *  Avahi: name|protocol, where name is the application name, and protocol the connection type.
     */
    void connect (std::string const &port);
    /// Accept a connection (only allowed on a listening socket).
    void accept (Glib::RefPtr <socket> sock);
    /// Get information about the other side of a connection.
    std::string get_peer_info (bool numeric = false) const;
    /// Get info about our side of the connection.
    std::string get_own_info (bool numeric = false) const;
    /// Schedule a function to be called when the socket is disconnected.
    disconnect_t signal_disconnect ();
    /// Disconnect the socket without reconnecting.
    void disconnect ();
  protected:
    /// Constructor.
    socket (Glib::RefPtr <Glib::MainContext> main);
    /// Destructor.
    virtual ~socket ();
  private:
    // Accept a new connection (called from main loop)
    void l_listen (listen_t cb);
    // transform a service or int to a real int
    static int l_service_to_port (std::string const &service);
    // Determine hostname and port from sockaddr_in
    std::string l_get_socket_info (struct sockaddr_in *addr,
				   bool numeric) const;
    // Finish disconnecting (read buffer is empty).
    void l_finish_disconnect ();
    // Callback function for the user: data has been read.
    read_t m_read;
    // Callback function for the user: socket is disconnected.
    disconnect_t m_disconnect;
    // Is this socket listening for connections?
    bool m_listener;
    // Remember the name of unix listensockets, to unlink them.
    std::string *m_name;
    // Callback for listensockets, called when a new connection is made.
    listen_t m_listen;
    // Avahi object for listening sockets.
    Glib::RefPtr <avahi> m_avahi;
  };
}

#endif // defined SHEVEK_SOCKET_HH
