/*
 * @(#)RubikGL.c
 *
 * Copyright 2003 - 2010  David A. Bagley, bagleyd@tux.org
 *
 * All rights reserved.
 *
 * Permission to use, copy, modify, and distribute this software and
 * its documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and
 * that both that copyright notice and this permission notice appear in
 * supporting documentation, and that the name of the author not be
 * used in advertising or publicity pertaining to distribution of the
 * software without specific, written prior permission.
 *
 * This program is distributed in the hope that it will be "useful",
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.
 */

/*-
 * Permission to use, copy, modify, and distribute this software and its
 * documentation for any purpose and without fee is hereby granted,
 * provided that the above copyright notice appear in all copies and that
 * both that copyright notice and this permission notice appear in
 * supporting documentation.
 *
 * This file is provided AS IS with no warranties of any kind.  The author
 * shall have no liability with respect to the infringement of copyrights,
 * trade secrets or any patents by this file or any part thereof.  In no
 *event will the author be liable for any lost revenue or profits or
 * other special, indirect and consequential damages.
 *
 * This mode shows an auto-solving rubik's cube "puzzle". If somebody
 * intends to make a game or something based on this code, please let me
 * know first, my e-mail address is provided in this comment. Marcelo.
 *
 * Thanks goes also to Brian Paul for making it possible and inexpensive
 * to use OpenGL at home.
 *
 * Since I'm not a native English speaker, my apologies for any grammatical
 * mistakes.
 *
 * My e-mail address is
 * mfvianna@centroin.com.br
 *
 * Marcelo F. Vianna (Jul-31-1997)
 */

#ifdef HAVE_OPENGL

/* Methods file for RubikGL */

#include "RubikP.h"
#include "RubikGLP.h"

static float front_shininess[] =
{60.0};
static float front_specular[] =
{0.7, 0.7, 0.7, 1.0};
static float ambient[] =
{0.0, 0.0, 0.0, 1.0};
static float diffuse[] =
{1.0, 1.0, 1.0, 1.0};
static float position0[] =
{1.0, 1.0, 1.0, 0.0};
static float position1[] =
{-1.0, -1.0, 1.0, 0.0};
static float lmodel_ambient[] =
{0.5, 0.5, 0.5, 1.0};
static float lmodel_twoside[] =
{GL_TRUE};

static float face_material[MAX_FACES + 2][4] =
{{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{1.0, 1.0, 1.0, 1.0},
{0.0, 0.0, 0.0, 1.0},
{0.0, 0.0, 0.0, 1.0}};

/*-
 * Examine facet 0 on each face, its 4 movements (well only 2 since the
 * other 2 will be opposites) and translate it into slice movements).
 * CW = DEEP Depth CCW == SHALLOW Depth with reference to faces 0, 1, and 2
 */
static RubikLoc rotateSlice[MAX_FACES][MAX_ORIENT / 2] =
{
	{
		{1, CCW},
		{2, CW},
	},
	{
		{2, CW},
		{0, CCW},
	},
	{
		{1, CCW},
		{0, CCW},
	},
	{
		{2, CCW},
		{0, CCW},
	},
	{
		{1, CCW},
		{2, CCW},
	},
	{
		{1, CCW},
		{0, CW},
	}
};

static int rotateDir[MAX_FACES][MAX_ORIENT] =
{
	{-1, -3, 1, 3},
	{-3, 2, 3, -2},
	{-1, 2, 1, -2},
	{3, 2, -3, -2},
	{-1, 3, 1, -3},
	{-1, -2, 1, 2}
};

/*-
 * Color labels mapping:
 * =====================
 *
 *             +-----------+
 *             |0-->       |
 *             ||          |
 *             |v  TOP(0)  |
 *             |   RED     |
 *             |          8|
 * +-----------+-----------+-----------+
 * |0-->       |0-->       |0-->       |
 * ||          ||          ||          |
 * |v  LEFT(1) |v FRONT(2) |v RIGHT(3) |
 * |   YELLOW  |  WHITE    |  GREEN    |
 * |          8|          8|          8|
 * +-----------+-----------+-----------+
 *             |0-->       |
 *             ||          |
 *             |v BOTTOM(4)|
 *             |  ORANGE   |
 *             |          8|
 *             +-----------+             +---+---+---+
 *             |0-->       |             | 0 | 1 | 2 |
 *             ||          |             |--xxxxx(N)-+
 *             |v  BACK(5) |             | 3 | 4 | 5 |
 *             |   BLUE    |             +---+---+---+
 *             |          8|             | 6 | 7 | 8 |
 *             +-----------+             +---+---+---+
 *
 *  Map to 3d
 *  FRONT  => X, Y
 *  BACK   => X, Y
 *  LEFT   => Z, Y
 *  RIGHT  => Z, Y
 *  TOP    => X, Z
 *  BOTTOM => X, Z
 */

static Boolean madeCurrent = False;

#ifdef WINVER
static HGLRC hRC = NULL;
#else
static GLXContext *glXContext = (GLXContext *) NULL;
static Boolean setValuesPuzzleGL(Widget current, Widget request, Widget renew);
static void resizePuzzleGL(RubikGLWidget w);
static void initializePuzzleGL(Widget request, Widget renew);
static void exposePuzzleGL(Widget renew, XEvent *event, Region region);
static void movePuzzleTl(RubikGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleTop(RubikGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleTr(RubikGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleLeft(RubikGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleRight(RubikGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleBl(RubikGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleBottom(RubikGLWidget w,
	XEvent *event, char **args, int nArgs);
static void movePuzzleBr(RubikGLWidget w,
	XEvent *event, char **args, int nArgs);

static char translationsGL[] =
"<KeyPress>q: Quit()\n\
 Ctrl<KeyPress>C: Quit()\n\
 <KeyPress>osfCancel: Hide()\n\
 <KeyPress>Escape: Hide()\n\
 <KeyPress>osfEscape: Hide()\n\
 Ctrl<KeyPress>[: Hide()\n\
 <KeyPress>0x1B: Hide()\n\
 <KeyPress>0x2E: Speed()\n\
 <KeyPress>0x3E: Speed()\n\
 <KeyPress>0x3C: Slow()\n\
 <KeyPress>0x2C: Slow()\n\
 Shift<KeyPress>2: Sound()\n\
 <KeyPress>F11: MoveCcw()\n\
 <KeyPress>KP_Divide: MoveCcw()\n\
 <KeyPress>R5: MoveCcw()\n\
 <KeyPress>Home: MoveTl()\n\
 <KeyPress>KP_7: MoveTl()\n\
 <KeyPress>R7: MoveTl()\n\
 <KeyPress>Up: MoveTop()\n\
 <KeyPress>osfUp: MoveTop()\n\
 <KeyPress>KP_Up: MoveTop()\n\
 <KeyPress>KP_8: MoveTop()\n\
 <KeyPress>R8: MoveTop()\n\
 <KeyPress>Prior: MoveTr()\n\
 <KeyPress>KP_9: MoveTr()\n\
 <KeyPress>R9: MoveTr()\n\
 <KeyPress>Left: MoveLeft()\n\
 <KeyPress>osfLeft: MoveLeft()\n\
 <KeyPress>KP_Left: MoveLeft()\n\
 <KeyPress>KP_4: MoveLeft()\n\
 <KeyPress>R10: MoveLeft()\n\
 <KeyPress>F12: MoveCw()\n\
 <KeyPress>Begin: MoveCw()\n\
 <KeyPress>KP_5: MoveCw()\n\
 <KeyPress>R11: MoveCw()\n\
 <KeyPress>Right: MoveRight()\n\
 <KeyPress>osfRight: MoveRight()\n\
 <KeyPress>KP_Right: MoveRight()\n\
 <KeyPress>KP_6: MoveRight()\n\
 <KeyPress>R12: MoveRight()\n\
 <KeyPress>End: MoveBl()\n\
 <KeyPress>KP_1: MoveBl()\n\
 <KeyPress>R13: MoveBl()\n\
 <KeyPress>Down: MoveBottom()\n\
 <KeyPress>osfDown: MoveBottom()\n\
 <KeyPress>KP_Down: MoveBottom()\n\
 <KeyPress>KP_2: MoveBottom()\n\
 <KeyPress>R14: MoveBottom()\n\
 <KeyPress>Next: MoveBr()\n\
 <KeyPress>KP_3: MoveBr()\n\
 <KeyPress>R15: MoveBr()\n\
 <Btn1Down>: Select()\n\
 <Btn1Up>: Release()\n\
 <Btn2Down>: PracticeMaybe()\n\
 <Btn2Down>(2+): Practice2()\n\
 <Btn3Down>: RandomizeMaybe()\n\
 <Btn3Down>(2+): Randomize2()\n\
 <Btn4Down>: MoveTop()\n\
 <Btn5Down>: MoveBottom()\n\
 <KeyPress>g: Get()\n\
 <KeyPress>w: Write()\n\
 <KeyPress>u: Undo()\n\
 <KeyPress>r: Redo()\n\
 <KeyPress>c: Clear()\n\
 <KeyPress>z: Randomize()\n\
 <KeyPress>s: Solve()\n\
 <KeyPress>p: Practice()\n\
 <KeyPress>i: Increment()\n\
 <KeyPress>d: Decrement()\n\
 <KeyPress>x: IncrementX()\n\
 <KeyPress>y: IncrementY()\n\
 <KeyPress>f: IncrementZ()\n\
 <KeyPress>o: Orientize()\n\
 <KeyPress>v: View()\n\
 <EnterWindow>: Enter()\n\
 <LeaveWindow>: Leave()";

static XtActionsRec actionsListGL[] =
{
	{(char *) "Quit", (XtActionProc) quitPuzzle},
	{(char *) "Hide", (XtActionProc) hidePuzzle},
	{(char *) "MoveCcw", (XtActionProc) movePuzzleCcw},
	{(char *) "MoveTl", (XtActionProc) movePuzzleTl},
	{(char *) "MoveTop", (XtActionProc) movePuzzleTop},
	{(char *) "MoveTr", (XtActionProc) movePuzzleTr},
	{(char *) "MoveLeft", (XtActionProc) movePuzzleLeft},
	{(char *) "MoveCw", (XtActionProc) movePuzzleCw},
	{(char *) "MoveRight", (XtActionProc) movePuzzleRight},
	{(char *) "MoveBl", (XtActionProc) movePuzzleBl},
	{(char *) "MoveBottom", (XtActionProc) movePuzzleBottom},
	{(char *) "MoveBr", (XtActionProc) movePuzzleBr},
	{(char *) "Select", (XtActionProc) selectPuzzle},
	{(char *) "Release", (XtActionProc) releasePuzzle},
	{(char *) "PracticeMaybe", (XtActionProc) practicePuzzleWithQuery},
	{(char *) "Practice2", (XtActionProc) practicePuzzleWithDoubleClick},
	{(char *) "RandomizeMaybe", (XtActionProc) randomizePuzzleWithQuery},
	{(char *) "Randomize2", (XtActionProc) randomizePuzzleWithDoubleClick},
	{(char *) "Get", (XtActionProc) getPuzzle},
	{(char *) "Write", (XtActionProc) writePuzzle},
	{(char *) "Undo", (XtActionProc) undoPuzzle},
	{(char *) "Redo", (XtActionProc) redoPuzzle},
	{(char *) "Clear", (XtActionProc) clearPuzzle},
	{(char *) "Randomize", (XtActionProc) randomizePuzzle},
	{(char *) "Solve", (XtActionProc) solvePuzzle},
	{(char *) "Practice", (XtActionProc) practicePuzzle},
	{(char *) "Increment", (XtActionProc) incrementPuzzle},
	{(char *) "Decrement", (XtActionProc) decrementPuzzle},
	{(char *) "IncrementX", (XtActionProc) incrementXPuzzle},
	{(char *) "IncrementY", (XtActionProc) incrementYPuzzle},
	{(char *) "IncrementZ", (XtActionProc) incrementZPuzzle},
	{(char *) "Orientize", (XtActionProc) orientizePuzzle},
	{(char *) "View", (XtActionProc) viewPuzzle},
	{(char *) "Speed", (XtActionProc) speedUpPuzzle},
	{(char *) "Slow", (XtActionProc) slowDownPuzzle},
	{(char *) "Sound", (XtActionProc) toggleSoundPuzzle},
	{(char *) "Enter", (XtActionProc) enterPuzzle},
	{(char *) "Leave", (XtActionProc) leavePuzzle}
};

static XtResource resourcesGL[] =
{
	{XtNwidth, XtCWidth, XtRDimension, sizeof (Dimension),
	 XtOffset(RubikWidget, core.width),
	 XtRString, (caddr_t) "250"},
	{XtNheight, XtCHeight, XtRDimension, sizeof (Dimension),
	 XtOffset(RubikWidget, core.height),
	 XtRString, (caddr_t) "400"},
	{XtNmono, XtCMono, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.mono),
	 XtRString, (caddr_t) "FALSE"},
	{XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.reverse),
	 XtRString, (caddr_t) "FALSE"},
	{XtNforeground, XtCForeground, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.foreground),
	 XtRString, (caddr_t) XtDefaultForeground},
	{XtNbackground, XtCBackground, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.background),
	 XtRString, (caddr_t) "#AEB2C3" /*XtDefaultBackground*/},
	{XtNframeColor, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.frameColor),
	 XtRString, (caddr_t) "Cyan" /*XtDefaultForeground*/},
	{XtNfaceColor0, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[0]),
	 XtRString, (caddr_t) "Red"},
	{XtNfaceColor1, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[1]),
	 XtRString, (caddr_t) "Yellow"},
	{XtNfaceColor2, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[2]),
	 XtRString, (caddr_t) "White"},
	{XtNfaceColor3, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[3]),
	 XtRString, (caddr_t) "Green"},
	{XtNfaceColor4, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[4]),
	 XtRString, (caddr_t) "Orange"},
	{XtNfaceColor5, XtCLabel, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.faceName[5]),
	 XtRString, (caddr_t) "Blue"},
	{XtNpieceBorder, XtCColor, XtRPixel, sizeof (Pixel),
	 XtOffset(RubikWidget, rubik.borderColor),
	 XtRString, (caddr_t) "gray25" /*XtDefaultForeground*/},
	{XtNdelay, XtCDelay, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.delay),
	 XtRString, (caddr_t) "10"},
	{XtNsound, XtCSound, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.sound),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmoveSound, XtCMoveSound, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.moveSound),
	 XtRString, (caddr_t) MOVESOUND},
	{XtNfont, XtCFont, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.font),
	 XtRString, (caddr_t) "9x15bold"},
	{XtNview, XtCView, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.view),
	 XtRString, (caddr_t) "1"},
	{XtNsizex, XtCSizeX, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizex),
	 XtRString, (caddr_t) "3"},	/* DEFAULTFACETS */
	{XtNsizey, XtCSizeY, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizey),
	 XtRString, (caddr_t) "3"},	/* DEFAULTFACETS */
	{XtNsizez, XtCSizeZ, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.sizez),
	 XtRString, (caddr_t) "3"},	/* DEFAULTFACETS */
	{XtNorient, XtCOrient, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.orient),
	 XtRString, (caddr_t) "FALSE"},	/* DEFAULT_ORIENT */
	{XtNpractice, XtCPractice, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.practice),
	 XtRString, (caddr_t) "TRUE"}, /* DEFAULT_PRACTICE */
	{XtNuserName, XtCUserName, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.userName),
	 XtRString, (caddr_t) ""},
	{XtNscoreFile, XtCScoreFile, XtRString, sizeof (String),
	 XtOffset(RubikWidget, rubik.scoreFile),
	 XtRString, (caddr_t) ""},
	{XtNscoreOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.scoreOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNversionOnly, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.versionOnly),
	 XtRString, (caddr_t) "FALSE"},
	{XtNmenu, XtCMenu, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.menu),
	 XtRString, (caddr_t) "999"}, /* ACTION_IGNORE */
	{XtNstart, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.started),
	 XtRString, (caddr_t) "FALSE"},
	{XtNcheat, XtCBoolean, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.cheat),
	 XtRString, (caddr_t) "FALSE"},
	{XtNface, XtCFace, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentFace),
	 XtRString, (caddr_t) "-1"},
	{XtNpos, XtCPos, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentPosition),
	 XtRString, (caddr_t) "-1"},
	{XtNdirection, XtCDirection, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentDirection),
	 XtRString, (caddr_t) "-1"},
	{XtNcontrol, XtCControl, XtRBoolean, sizeof (Boolean),
	 XtOffset(RubikWidget, rubik.currentControl),
	 XtRString, (caddr_t) "FALSE"},
	{XtNfast, XtCFast, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.currentFast),
	 XtRString, (caddr_t) "1"},
	{XtNpixmapSize, XtCPixmapSize, XtRInt, sizeof (int),
	 XtOffset(RubikWidget, rubik.pixmapSize),
	 XtRString, (caddr_t) "64"},
	{XtNselectCallback, XtCCallback, XtRCallback, sizeof (caddr_t),
	 XtOffset(RubikWidget, rubik.select),
	 XtRCallback, (caddr_t) NULL}
};

RubikGLClassRec rubikGLClassRec =
{
	{
		(WidgetClass) & rubikClassRec,	/* superclass */
		(char *) "RubikGL",	/* class name */
		sizeof (RubikGLRec),	/* widget size */
		NULL,		/* class initialize */
		NULL,		/* class part initialize */
		FALSE,		/* class inited */
		(XtInitProc) initializePuzzleGL,	/* initialize */
		NULL,		/* initialize hook */
		XtInheritRealize,	/* realize */
		actionsListGL,	/* actions */
		XtNumber(actionsListGL),	/* num actions */
		resourcesGL,	/* resources */
		XtNumber(resourcesGL),	/* num resources */
		NULLQUARK,	/* xrm class */
		TRUE,		/* compress motion */
		TRUE,		/* compress exposure */
		TRUE,		/* compress enterleave */
		TRUE,		/* visible interest */
		NULL,		/* destroy */
		(XtWidgetProc) resizePuzzleGL,	/* resize */
		(XtExposeProc) exposePuzzleGL,	/* expose */
		(XtSetValuesFunc) setValuesPuzzleGL,	/* set values */
		NULL,		/* set values hook */
		XtInheritSetValuesAlmost,	/* set values almost */
		NULL,		/* get values hook */
		XtInheritAcceptFocus,	/* accept focus */
		XtVersion,	/* version */
		NULL,		/* callback private */
		translationsGL,	/* tm table */
		NULL,		/* query geometry */
		NULL,		/* display accelerator */
		NULL		/* extension */
	},
	{
		0		/* ignore */
	},
	{
		0		/* ignore */
	}
};

WidgetClass rubikGLWidgetClass = (WidgetClass) & rubikGLClassRec;
#endif

static RowNext rotateToRow[MAX_FACES] = /* CW to min face */
{
	{1, LEFT, TOP},
	{0, BOTTOM, RIGHT},
	{0, RIGHT, BOTTOM},
	{0, TOP, LEFT},
	{1, RIGHT, BOTTOM},
	{0, LEFT, TOP}
};

#ifdef DEBUG
void
printCube(RubikGLWidget w)
{
	int face, position, sizeOfRow, sizeOfColumn;

	for (face = 0; face < MAX_FACES; face++) {
		faceSizes(w, face, &sizeOfRow, &sizeOfColumn);
		for (position = 0; position < sizeOfRow * sizeOfColumn; position++) {
			(void) printf("%d %d  ",
				w->rubik.cubeLoc[face][position].face,
				w->rubik.cubeLoc[face][position].rotation);
			if (!((position + 1) % sizeOfRow))
				(void) printf("\n");
		}
		(void) printf("\n");
	}
	(void) printf("\n");
}

#endif

static void
pickColor(RubikGLWidget w, int C, int mono, float *material)
{
#ifdef WINVER
	struct tagColor {
		int red, green, blue;
	} color;
#define MAX_INTENSITY 0xFF
#else
	XColor color;
#define MAX_INTENSITY 0xFFFF
#endif

	switch (C) {
	case TOP_FACE:
	case LEFT_FACE:
	case FRONT_FACE:
	case RIGHT_FACE:
	case BOTTOM_FACE:
	case BACK_FACE:
#ifdef WINVER
		color.red = GetRValue(w->rubik.faceGC[C]);
		color.green = GetGValue(w->rubik.faceGC[C]);
		color.blue = GetBValue(w->rubik.faceGC[C]);
#else
		color.pixel = w->rubik.faceColor[C];
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	case BORDER_FACE:
#ifdef WINVER
		color.red = GetRValue(w->rubik.borderGC);
		color.green = GetGValue(w->rubik.borderGC);
		color.blue = GetBValue(w->rubik.borderGC);
#else
		color.pixel = w->rubik.borderColor;
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	case NO_FACE:
	default:
#ifdef WINVER
		color.red = GetRValue(w->rubik.inverseGC);
		color.green = GetGValue(w->rubik.inverseGC);
		color.blue = GetBValue(w->rubik.inverseGC);
#else
		color.pixel = w->rubik.background;
		XQueryColor(XtDisplay(w), DefaultColormapOfScreen(XtScreen(w)),
			&color);
#endif
		break;
	}
#ifdef DEBUG
	(void) printf("i%d %d %d\n", color.red, color.green, color.blue);
#endif
	if (mono) {
		/* really GrayScale */
		float intensity = 0.3 * color.red +
			0.59 * color.green + 0.11 * color.blue;

#ifdef DEBUG
		(void) printf("m%g\n", intensity);
#endif
		material[0] = (float) intensity / MAX_INTENSITY;
		material[1] = material[0];
		material[2] = material[0];
	} else {
		material[0] = (float) color.red / MAX_INTENSITY;
		material[1] = (float) color.green / MAX_INTENSITY;
		material[2] = (float) color.blue / MAX_INTENSITY;
	}
#ifdef DEBUG
	(void) printf("f%g %g %g\n", material[0], material[1], material[2]);
#endif
}

typedef float Matrix[3][3];
typedef float Vector[3];

static void
matrixMatrix(Matrix a, Matrix b, Matrix x)
{
	int i, j;

	for (i = 0; i < 3; i++) {
		for (j = 0; j < 3; j++) {
			x[i][j] = a[i][0] * b[0][j] +
				a[i][1] * b[1][j] +
				a[i][2] * b[2][j];
		}
	}
}

static void
matrixVector(Matrix a, Vector v, Vector x)
{
	int i;

	for (i = 0; i < 3; i++) {
		x[i] = a[i][0] * v[0] +
			a[i][1] * v[1] +
			a[i][2] * v[2];
	}
}

static void
setMatrix(float theta, float x, float y, float z, Matrix a)
{
	double s = sin(theta * M_PI / 180.0);
	double c = cos(theta * M_PI / 180.0);

	if (x != 0.0 && y == 0.0 && z == 0.0) {
		a[0][0] = 1.0, a[0][1] = 0.0; a[0][2] = 0.0;
		a[1][0] = 0.0, a[1][1] = c; a[1][2] = -s;
		a[2][0] = 0.0, a[2][1] = s; a[2][2] = c;
	} else if (y != 0.0 && x == 0.0 && z == 0.0) {
		a[0][0] = c, a[0][1] = 0.0; a[0][2] = s;
		a[1][0] = 0.0, a[1][1] = 1.0; a[1][2] = 0.0;
		a[2][0] = -s, a[2][1] = 0.0; a[2][2] = c;
	} else if (z != 0.0 && x == 0.0 && y == 0.0) {
		a[0][0] = c, a[0][1] = -s; a[0][2] = 0.0;
		a[1][0] = s, a[1][1] = c; a[1][2] = 0.0;
		a[2][0] = 0.0, a[2][1] = 0.0; a[2][2] = 1.0;
	}
}

static Boolean
drawStickerlessCubit(void)
{
	glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
		face_material[BORDER_FACE]);
	glBegin(GL_QUADS);
	/* Put sticker here */
	glNormal3f(0.0, 0.0, 1.0);
	glVertex3f(-CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glNormal3f(0.0, 0.0, -1.0);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, -CUBE_ROUND, -CUBE_LENGTH);
	glNormal3f(-1.0, 0.0, 0.0);
	glVertex3f(-CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, -CUBE_ROUND, -CUBE_ROUND);
	glNormal3f(1.0, 0.0, 0.0);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glNormal3f(0.0, -1.0, 0.0);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, -CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, -CUBE_LENGTH, -CUBE_ROUND);
	glNormal3f(0.0, 1.0, 0.0);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);

	/* Edges of cubit */
	glNormal3f(-1.0, -1.0, 0.0);
	glVertex3f(-CUBE_ROUND, -CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, -CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, -CUBE_ROUND, -CUBE_ROUND);
	glNormal3f(1.0, 1.0, 0.0);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glNormal3f(-1.0, 1.0, 0.0);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glNormal3f(1.0, -1.0, 0.0);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, -CUBE_ROUND);
	glNormal3f(0.0, -1.0, -1.0);
	glVertex3f(-CUBE_ROUND, -CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, -CUBE_LENGTH, -CUBE_ROUND);
	glNormal3f(0.0, 1.0, 1.0);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glNormal3f(0.0, -1.0, 1.0);
	glVertex3f(-CUBE_ROUND, -CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glNormal3f(0.0, 1.0, -1.0);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, -CUBE_LENGTH);
	glNormal3f(-1.0, 0.0, -1.0);
	glVertex3f(-CUBE_LENGTH, -CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, -CUBE_ROUND, -CUBE_LENGTH);
	glNormal3f(1.0, 0.0, 1.0);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glNormal3f(1.0, 0.0, -1.0);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, -CUBE_ROUND);
	glNormal3f(-1.0, 0.0, 1.0);
	glVertex3f(-CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glEnd();

	glBegin(GL_TRIANGLES);
	/* Corners of cubit */
	glNormal3f(1.0, 1.0, 1.0);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glNormal3f(-1.0, -1.0, -1.0);
	glVertex3f(-CUBE_ROUND, -CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, -CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, -CUBE_ROUND, -CUBE_LENGTH);
	glNormal3f(-1.0, 1.0, 1.0);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, CUBE_ROUND);
	glNormal3f(1.0, -1.0, -1.0);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, -CUBE_LENGTH);
	glNormal3f(1.0, -1.0, 1.0);
	glVertex3f(CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, -CUBE_LENGTH, CUBE_ROUND);
	glVertex3f(CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glNormal3f(-1.0, 1.0, -1.0);
	glVertex3f(-CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, CUBE_ROUND, -CUBE_LENGTH);
	glNormal3f(-1.0, -1.0, 1.0);
	glVertex3f(-CUBE_ROUND, -CUBE_ROUND, CUBE_LENGTH);
	glVertex3f(-CUBE_LENGTH, -CUBE_ROUND, CUBE_ROUND);
	glVertex3f(-CUBE_ROUND, -CUBE_LENGTH, CUBE_ROUND);
	glNormal3f(1.0, 1.0, -1.0);
	glVertex3f(CUBE_LENGTH, CUBE_ROUND, -CUBE_ROUND);
	glVertex3f(CUBE_ROUND, CUBE_ROUND, -CUBE_LENGTH);
	glVertex3f(CUBE_ROUND, CUBE_LENGTH, -CUBE_ROUND);
	glEnd();
	return True;
}

static Boolean
drawCubit(RubikGLWidget w,
		int back, int front, int left, int right, int bottom, int top)
{
	char *buf1, *buf2;

	if (!drawStickerlessCubit())
		return False;
	if (back != NO_POSITION) { /* Blue */
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[(int) w->rubik.cubeLoc[BACK_FACE][back].face]);
		glNormal3f(0.0, 0.0, -1.0);
		glVertex3f(-STICKER_SHORT, STICKER_LONG, -STICKER_DEPTH);
		glVertex3f(STICKER_SHORT, STICKER_LONG, -STICKER_DEPTH);
		glVertex3f(STICKER_LONG, STICKER_SHORT, -STICKER_DEPTH);
		glVertex3f(STICKER_LONG, -STICKER_SHORT, -STICKER_DEPTH);
		glVertex3f(STICKER_SHORT, -STICKER_LONG, -STICKER_DEPTH);
		glVertex3f(-STICKER_SHORT, -STICKER_LONG, -STICKER_DEPTH);
		glVertex3f(-STICKER_LONG, -STICKER_SHORT, -STICKER_DEPTH);
		glVertex3f(-STICKER_LONG, STICKER_SHORT, -STICKER_DEPTH);
		glEnd();
		if (w->rubik.currentFace == BACK_FACE &&
				w->rubik.currentPosition == back) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 0.0, -1.0);
			glVertex3f(-SELECT_SHORT, SELECT_LONG, -SELECT_DEPTH);
			glVertex3f(SELECT_SHORT, SELECT_LONG, -SELECT_DEPTH);
			glVertex3f(SELECT_LONG, SELECT_SHORT, -SELECT_DEPTH);
			glVertex3f(SELECT_LONG, -SELECT_SHORT, -SELECT_DEPTH);
			glVertex3f(SELECT_SHORT, -SELECT_LONG, -SELECT_DEPTH);
			glVertex3f(-SELECT_SHORT, -SELECT_LONG, -SELECT_DEPTH);
			glVertex3f(-SELECT_LONG, -SELECT_SHORT, -SELECT_DEPTH);
			glVertex3f(-SELECT_LONG, SELECT_SHORT, -SELECT_DEPTH);
			glEnd();
		}
		if (w->rubik.orient) {
			int rotation = w->rubik.cubeLoc[BACK_FACE][back].rotation;

			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 0.0, -1.0);
			switch (rotation) {
			case 0:
				glVertex3f(CUT, -CUT, -CUT_DEPTH);
				glVertex3f(CUT, -STICKER_LONG, -CUT_DEPTH);
				glVertex3f(-CUT, -STICKER_LONG, -CUT_DEPTH);
				glVertex3f(-CUT, -CUT, -CUT_DEPTH);
				break;
			case 1:
				glVertex3f(CUT, CUT, -CUT_DEPTH);
				glVertex3f(STICKER_LONG, CUT, -CUT_DEPTH);
				glVertex3f(STICKER_LONG, -CUT, -CUT_DEPTH);
				glVertex3f(CUT, -CUT, -CUT_DEPTH);
				break;
			case 2:
				glVertex3f(-CUT, CUT, -CUT_DEPTH);
				glVertex3f(-CUT, STICKER_LONG, -CUT_DEPTH);
				glVertex3f(CUT, STICKER_LONG, -CUT_DEPTH);
				glVertex3f(CUT, CUT, -CUT_DEPTH);
				break;
			case 3:
				glVertex3f(-CUT, -CUT, -CUT_DEPTH);
				glVertex3f(-STICKER_LONG, -CUT, -CUT_DEPTH);
				glVertex3f(-STICKER_LONG, CUT, -CUT_DEPTH);
				glVertex3f(-CUT, CUT, -CUT_DEPTH);
				break;
			default:
				intCat(&buf1, "drawCubit: face ", BACK_FACE);
				stringCat(&buf2, buf1, ", rotation ");
				free(buf1);
				intCat(&buf1, buf2, rotation);
				free(buf2);
				DISPLAY_WARNING(buf1);
				free(buf1);
			}
			glEnd();
		}
	}
	if (front != NO_POSITION) { /* White */
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[(int) w->rubik.cubeLoc[FRONT_FACE][front].face]);
		glNormal3f(0.0, 0.0, 1.0);
		glVertex3f(-STICKER_SHORT, -STICKER_LONG, STICKER_DEPTH);
		glVertex3f(STICKER_SHORT, -STICKER_LONG, STICKER_DEPTH);
		glVertex3f(STICKER_LONG, -STICKER_SHORT, STICKER_DEPTH);
		glVertex3f(STICKER_LONG, STICKER_SHORT, STICKER_DEPTH);
		glVertex3f(STICKER_SHORT, STICKER_LONG, STICKER_DEPTH);
		glVertex3f(-STICKER_SHORT, STICKER_LONG, STICKER_DEPTH);
		glVertex3f(-STICKER_LONG, STICKER_SHORT, STICKER_DEPTH);
		glVertex3f(-STICKER_LONG, -STICKER_SHORT, STICKER_DEPTH);
		glEnd();
		if (w->rubik.currentFace == FRONT_FACE &&
				w->rubik.currentPosition == front) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 0.0, 1.0);
			glVertex3f(-SELECT_SHORT, -SELECT_LONG, SELECT_DEPTH);
			glVertex3f(SELECT_SHORT, -SELECT_LONG, SELECT_DEPTH);
			glVertex3f(SELECT_LONG, -SELECT_SHORT, SELECT_DEPTH);
			glVertex3f(SELECT_LONG, SELECT_SHORT, SELECT_DEPTH);
			glVertex3f(SELECT_SHORT, SELECT_LONG, SELECT_DEPTH);
			glVertex3f(-SELECT_SHORT, SELECT_LONG, SELECT_DEPTH);
			glVertex3f(-SELECT_LONG, SELECT_SHORT, SELECT_DEPTH);
			glVertex3f(-SELECT_LONG, -SELECT_SHORT, SELECT_DEPTH);
			glEnd();
		}
		if (w->rubik.orient) {
			int rotation = w->rubik.cubeLoc[FRONT_FACE][front].rotation;

			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 0.0, 1.0);
			switch (rotation) {
			case 0:
				glVertex3f(CUT, CUT, CUT_DEPTH);
				glVertex3f(CUT, STICKER_LONG, CUT_DEPTH);
				glVertex3f(-CUT, STICKER_LONG, CUT_DEPTH);
				glVertex3f(-CUT, CUT, CUT_DEPTH);
				break;
			case 1:
				glVertex3f(CUT, -CUT, CUT_DEPTH);
				glVertex3f(STICKER_LONG, -CUT, CUT_DEPTH);
				glVertex3f(STICKER_LONG, CUT, CUT_DEPTH);
				glVertex3f(CUT, CUT, CUT_DEPTH);
				break;
			case 2:
				glVertex3f(-CUT, -CUT, CUT_DEPTH);
				glVertex3f(-CUT, -STICKER_LONG, CUT_DEPTH);
				glVertex3f(CUT, -STICKER_LONG, CUT_DEPTH);
				glVertex3f(CUT, -CUT, CUT_DEPTH);
				break;
			case 3:
				glVertex3f(-CUT, CUT, CUT_DEPTH);
				glVertex3f(-STICKER_LONG, CUT, CUT_DEPTH);
				glVertex3f(-STICKER_LONG, -CUT, CUT_DEPTH);
				glVertex3f(-CUT, -CUT, CUT_DEPTH);
				break;
			default:
				intCat(&buf1, "drawCubit: face ", FRONT_FACE);
				stringCat(&buf2, buf1, ", rotation ");
				free(buf1);
				intCat(&buf1, buf2, rotation);
				free(buf2);
				DISPLAY_WARNING(buf1);
				free(buf1);
			}
			glEnd();
		}
	}
	if (left != NO_POSITION) { /* Yellow */
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[(int) w->rubik.cubeLoc[LEFT_FACE][left].face]);
		glNormal3f(-1.0, 0.0, 0.0);
		glVertex3f(-STICKER_DEPTH, -STICKER_SHORT, STICKER_LONG);
		glVertex3f(-STICKER_DEPTH, STICKER_SHORT, STICKER_LONG);
		glVertex3f(-STICKER_DEPTH, STICKER_LONG, STICKER_SHORT);
		glVertex3f(-STICKER_DEPTH, STICKER_LONG, -STICKER_SHORT);
		glVertex3f(-STICKER_DEPTH, STICKER_SHORT, -STICKER_LONG);
		glVertex3f(-STICKER_DEPTH, -STICKER_SHORT, -STICKER_LONG);
		glVertex3f(-STICKER_DEPTH, -STICKER_LONG, -STICKER_SHORT);
		glVertex3f(-STICKER_DEPTH, -STICKER_LONG, STICKER_SHORT);
		glEnd();
		if (w->rubik.currentFace == LEFT_FACE &&
				w->rubik.currentPosition == left) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(-1.0, 0.0, 0.0);
			glVertex3f(-SELECT_DEPTH, -SELECT_SHORT, SELECT_LONG);
			glVertex3f(-SELECT_DEPTH, SELECT_SHORT, SELECT_LONG);
			glVertex3f(-SELECT_DEPTH, SELECT_LONG, SELECT_SHORT);
			glVertex3f(-SELECT_DEPTH, SELECT_LONG, -SELECT_SHORT);
			glVertex3f(-SELECT_DEPTH, SELECT_SHORT, -SELECT_LONG);
			glVertex3f(-SELECT_DEPTH, -SELECT_SHORT, -SELECT_LONG);
			glVertex3f(-SELECT_DEPTH, -SELECT_LONG, -SELECT_SHORT);
			glVertex3f(-SELECT_DEPTH, -SELECT_LONG, SELECT_SHORT);
			glEnd();
		}
		if (w->rubik.orient) {
			int rotation = w->rubik.cubeLoc[LEFT_FACE][left].rotation;

			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(-1.0, 0.0, 0.0);
			switch (rotation) {
			case 0:
				glVertex3f(-CUT_DEPTH, CUT, CUT);
				glVertex3f(-CUT_DEPTH, STICKER_LONG, CUT);
				glVertex3f(-CUT_DEPTH, STICKER_LONG, -CUT);
				glVertex3f(-CUT_DEPTH, CUT, -CUT);
				break;
			case 1:
				glVertex3f(-CUT_DEPTH, -CUT, CUT);
				glVertex3f(-CUT_DEPTH, -CUT, STICKER_LONG);
				glVertex3f(-CUT_DEPTH, CUT, STICKER_LONG);
				glVertex3f(-CUT_DEPTH, CUT, CUT);
				break;
			case 2:
				glVertex3f(-CUT_DEPTH, -CUT, -CUT);
				glVertex3f(-CUT_DEPTH, -STICKER_LONG, -CUT);
				glVertex3f(-CUT_DEPTH, -STICKER_LONG, CUT);
				glVertex3f(-CUT_DEPTH, -CUT, CUT);
				break;
			case 3:
				glVertex3f(-CUT_DEPTH, CUT, -CUT);
				glVertex3f(-CUT_DEPTH, CUT, -STICKER_LONG);
				glVertex3f(-CUT_DEPTH, -CUT, -STICKER_LONG);
				glVertex3f(-CUT_DEPTH, -CUT, -CUT);
				break;
			default:
				intCat(&buf1, "drawCubit: face ", LEFT_FACE);
				stringCat(&buf2, buf1, ", rotation ");
				free(buf1);
				intCat(&buf1, buf2, rotation);
				free(buf2);
				DISPLAY_WARNING(buf1);
				free(buf1);
			}
			glEnd();
		}
	}
	if (right != NO_POSITION) { /* Green */
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[(int) w->rubik.cubeLoc[RIGHT_FACE][right].face]);
		glNormal3f(1.0, 0.0, 0.0);
		glVertex3f(STICKER_DEPTH, -STICKER_SHORT, -STICKER_LONG);
		glVertex3f(STICKER_DEPTH, STICKER_SHORT, -STICKER_LONG);
		glVertex3f(STICKER_DEPTH, STICKER_LONG, -STICKER_SHORT);
		glVertex3f(STICKER_DEPTH, STICKER_LONG, STICKER_SHORT);
		glVertex3f(STICKER_DEPTH, STICKER_SHORT, STICKER_LONG);
		glVertex3f(STICKER_DEPTH, -STICKER_SHORT, STICKER_LONG);
		glVertex3f(STICKER_DEPTH, -STICKER_LONG, STICKER_SHORT);
		glVertex3f(STICKER_DEPTH, -STICKER_LONG, -STICKER_SHORT);
		glEnd();
		if (w->rubik.currentFace == RIGHT_FACE &&
				w->rubik.currentPosition == right) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(1.0, 0.0, 0.0);
			glVertex3f(SELECT_DEPTH, -SELECT_SHORT, -SELECT_LONG);
			glVertex3f(SELECT_DEPTH, SELECT_SHORT, -SELECT_LONG);
			glVertex3f(SELECT_DEPTH, SELECT_LONG, -SELECT_SHORT);
			glVertex3f(SELECT_DEPTH, SELECT_LONG, SELECT_SHORT);
			glVertex3f(SELECT_DEPTH, SELECT_SHORT, SELECT_LONG);
			glVertex3f(SELECT_DEPTH, -SELECT_SHORT, SELECT_LONG);
			glVertex3f(SELECT_DEPTH, -SELECT_LONG, SELECT_SHORT);
			glVertex3f(SELECT_DEPTH, -SELECT_LONG, -SELECT_SHORT);
			glEnd();
		}
		if (w->rubik.orient) {
			int rotation = w->rubik.cubeLoc[RIGHT_FACE][right].rotation;

			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(1.0, 0.0, 0.0);
			switch (rotation) {
			case 0:
				glVertex3f(CUT_DEPTH, CUT, -CUT);
				glVertex3f(CUT_DEPTH, STICKER_LONG, -CUT);
				glVertex3f(CUT_DEPTH, STICKER_LONG, CUT);
				glVertex3f(CUT_DEPTH, CUT, CUT);
				break;
			case 1:
				glVertex3f(CUT_DEPTH, -CUT, -CUT);
				glVertex3f(CUT_DEPTH, -CUT, -STICKER_LONG);
				glVertex3f(CUT_DEPTH, CUT, -STICKER_LONG);
				glVertex3f(CUT_DEPTH, CUT, -CUT);
				break;
			case 2:
				glVertex3f(CUT_DEPTH, -CUT, CUT);
				glVertex3f(CUT_DEPTH, -STICKER_LONG, CUT);
				glVertex3f(CUT_DEPTH, -STICKER_LONG, -CUT);
				glVertex3f(CUT_DEPTH, -CUT, -CUT);
				break;
			case 3:
				glVertex3f(CUT_DEPTH, CUT, CUT);
				glVertex3f(CUT_DEPTH, CUT, STICKER_LONG);
				glVertex3f(CUT_DEPTH, -CUT, STICKER_LONG);
				glVertex3f(CUT_DEPTH, -CUT, CUT);
				break;
			default:
				intCat(&buf1, "drawCubit: face ", RIGHT_FACE);
				stringCat(&buf2, buf1, ", rotation ");
				free(buf1);
				intCat(&buf1, buf2, rotation);
				free(buf2);
				DISPLAY_WARNING(buf1);
				free(buf1);
			}
			glEnd();
		}
	}
	if (bottom != NO_POSITION) { /* Orange */
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[(int) w->rubik.cubeLoc[BOTTOM_FACE][bottom].face]);
		glNormal3f(0.0, -1.0, 0.0);
		glVertex3f(STICKER_LONG, -STICKER_DEPTH, -STICKER_SHORT);
		glVertex3f(STICKER_LONG, -STICKER_DEPTH, STICKER_SHORT);
		glVertex3f(STICKER_SHORT, -STICKER_DEPTH, STICKER_LONG);
		glVertex3f(-STICKER_SHORT, -STICKER_DEPTH, STICKER_LONG);
		glVertex3f(-STICKER_LONG, -STICKER_DEPTH, STICKER_SHORT);
		glVertex3f(-STICKER_LONG, -STICKER_DEPTH, -STICKER_SHORT);
		glVertex3f(-STICKER_SHORT, -STICKER_DEPTH, -STICKER_LONG);
		glVertex3f(STICKER_SHORT, -STICKER_DEPTH, -STICKER_LONG);
		glEnd();
		if (w->rubik.currentFace == BOTTOM_FACE &&
				w->rubik.currentPosition == bottom) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, -1.0, 0.0);
			glVertex3f(SELECT_LONG, -SELECT_DEPTH, -SELECT_SHORT);
			glVertex3f(SELECT_LONG, -SELECT_DEPTH, SELECT_SHORT);
			glVertex3f(SELECT_SHORT, -SELECT_DEPTH, SELECT_LONG);
			glVertex3f(-SELECT_SHORT, -SELECT_DEPTH, SELECT_LONG);
			glVertex3f(-SELECT_LONG, -SELECT_DEPTH, SELECT_SHORT);
			glVertex3f(-SELECT_LONG, -SELECT_DEPTH, -SELECT_SHORT);
			glVertex3f(-SELECT_SHORT, -SELECT_DEPTH, -SELECT_LONG);
			glVertex3f(SELECT_SHORT, -SELECT_DEPTH, -SELECT_LONG);
			glEnd();
		}
		if (w->rubik.orient) {
			int rotation = w->rubik.cubeLoc[BOTTOM_FACE][bottom].rotation;

			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, -1.0, 0.0);
			switch (rotation) {
			case 0:
				glVertex3f(CUT, -CUT_DEPTH, CUT);
				glVertex3f(CUT, -CUT_DEPTH, STICKER_LONG);
				glVertex3f(-CUT, -CUT_DEPTH, STICKER_LONG);
				glVertex3f(-CUT, -CUT_DEPTH, CUT);
				break;
			case 1:
				glVertex3f(CUT, -CUT_DEPTH, -CUT);
				glVertex3f(STICKER_LONG, -CUT_DEPTH, -CUT);
				glVertex3f(STICKER_LONG, -CUT_DEPTH, CUT);
				glVertex3f(CUT, -CUT_DEPTH, CUT);
				break;
			case 2:
				glVertex3f(-CUT, -CUT_DEPTH, -CUT);
				glVertex3f(-CUT, -CUT_DEPTH, -STICKER_LONG);
				glVertex3f(CUT, -CUT_DEPTH, -STICKER_LONG);
				glVertex3f(CUT, -CUT_DEPTH, -CUT);
				break;
			case 3:
				glVertex3f(-CUT, -CUT_DEPTH, CUT);
				glVertex3f(-STICKER_LONG, -CUT_DEPTH, CUT);
				glVertex3f(-STICKER_LONG, -CUT_DEPTH, -CUT);
				glVertex3f(-CUT, -CUT_DEPTH, -CUT);
				break;
			default:
				intCat(&buf1, "drawCubit: face ", BOTTOM_FACE);
				stringCat(&buf2, buf1, ", rotation ");
				free(buf1);
				intCat(&buf1, buf2, rotation);
				free(buf2);
				DISPLAY_WARNING(buf1);
				free(buf1);
			}
			glEnd();
		}
	}
	if (top != NO_POSITION) { /* Red */
		glBegin(GL_POLYGON);
		glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
			face_material[(int) w->rubik.cubeLoc[TOP_FACE][top].face]);
		glNormal3f(0.0, 1.0, 0.0);
		glVertex3f(-STICKER_LONG, STICKER_DEPTH, -STICKER_SHORT);
		glVertex3f(-STICKER_LONG, STICKER_DEPTH, STICKER_SHORT);
		glVertex3f(-STICKER_SHORT, STICKER_DEPTH, STICKER_LONG);
		glVertex3f(STICKER_SHORT, STICKER_DEPTH, STICKER_LONG);
		glVertex3f(STICKER_LONG, STICKER_DEPTH, STICKER_SHORT);
		glVertex3f(STICKER_LONG, STICKER_DEPTH, -STICKER_SHORT);
		glVertex3f(STICKER_SHORT, STICKER_DEPTH, -STICKER_LONG);
		glVertex3f(-STICKER_SHORT, STICKER_DEPTH, -STICKER_LONG);
		glEnd();
		if (w->rubik.currentFace == TOP_FACE &&
				w->rubik.currentPosition == top) {
			glBegin(GL_POLYGON);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 1.0, 0.0);
			glVertex3f(-SELECT_LONG, SELECT_DEPTH, -SELECT_SHORT);
			glVertex3f(-SELECT_LONG, SELECT_DEPTH, SELECT_SHORT);
			glVertex3f(-SELECT_SHORT, SELECT_DEPTH, SELECT_LONG);
			glVertex3f(SELECT_SHORT, SELECT_DEPTH, SELECT_LONG);
			glVertex3f(SELECT_LONG, SELECT_DEPTH, SELECT_SHORT);
			glVertex3f(SELECT_LONG, SELECT_DEPTH, -SELECT_SHORT);
			glVertex3f(SELECT_SHORT, SELECT_DEPTH, -SELECT_LONG);
			glVertex3f(-SELECT_SHORT, SELECT_DEPTH, -SELECT_LONG);
			glEnd();
		}
		if (w->rubik.orient) {
			int rotation = w->rubik.cubeLoc[TOP_FACE][top].rotation;

			glBegin(GL_QUADS);
			glMaterialfv(GL_FRONT_AND_BACK, GL_DIFFUSE,
				face_material[BORDER_FACE]);
			glNormal3f(0.0, 1.0, 0.0);
			switch (rotation) {
			case 0:
				glVertex3f(CUT, CUT_DEPTH, -CUT);
				glVertex3f(CUT, CUT_DEPTH, -STICKER_LONG);
				glVertex3f(-CUT, CUT_DEPTH, -STICKER_LONG);
				glVertex3f(-CUT, CUT_DEPTH, -CUT);
				break;
			case 1:
				glVertex3f(CUT, CUT_DEPTH, CUT);
				glVertex3f(STICKER_LONG, CUT_DEPTH, CUT);
				glVertex3f(STICKER_LONG, CUT_DEPTH, -CUT);
				glVertex3f(CUT, CUT_DEPTH, -CUT);
				break;
			case 2:
				glVertex3f(-CUT, CUT_DEPTH, CUT);
				glVertex3f(-CUT, CUT_DEPTH, STICKER_LONG);
				glVertex3f(CUT, CUT_DEPTH, STICKER_LONG);
				glVertex3f(CUT, CUT_DEPTH, CUT);
				break;
			case 3:
				glVertex3f(-CUT, CUT_DEPTH, -CUT);
				glVertex3f(-STICKER_LONG, CUT_DEPTH, -CUT);
				glVertex3f(-STICKER_LONG, CUT_DEPTH, CUT);
				glVertex3f(-CUT, CUT_DEPTH, CUT);
				break;
			default:
				intCat(&buf1, "drawCubit: face ", TOP_FACE);
				stringCat(&buf2, buf1, ", rotation ");
				free(buf1);
				intCat(&buf1, buf2, rotation);
				free(buf2);
				DISPLAY_WARNING(buf1);
				free(buf1);
			}
			glEnd();
		}
	}
	return True;
}

/* Convert move to weird general notation */
static void
convertMove(RubikGLWidget w, RubikMove move, RubikSlice * slice)
{
	RubikLoc plane;
	int sizeOfRow, sizeOfColumn;

	plane = rotateSlice[(int) move.face][move.direction % 2];
	(*slice).face = plane.face;
	(*slice).rotation = plane.rotation;
	faceSizes((RubikWidget) w, move.face, &sizeOfRow, &sizeOfColumn);
	if (plane.face == 1 || /* VERTICAL */
			(plane.face == 2 &&
			(move.face == 1 || move.face == 3))) {
		if ((*slice).rotation == CW)
			(*slice).depth = sizeOfRow - 1 - move.position %
				sizeOfRow;
		else
			(*slice).depth = move.position % sizeOfRow;
	} else { /* (plane.face == 0 || *//* HORIZONTAL *//*
			(plane.face == 2 &&
			(move.face == 0 || move.face == 4))) */
		if ((*slice).rotation == CW)
			(*slice).depth = sizeOfColumn - 1 - move.position /
				sizeOfRow;
		else
			(*slice).depth = move.position / sizeOfRow;
	}
	/* If (*slice).depth = 0 then face 0, face 1, or face 2 moves */
	if (move.direction / 2)
		(*slice).rotation = ((*slice).rotation == CW) ? CCW : CW;
}

/* Assume the size is at least 2, or its just not challenging... */
static Boolean
drawCube(RubikGLWidget w)
{
#define S1 1
#define SX ((GLint)S1*(MAX_SIZEX-1))
#define SY ((GLint)S1*(MAX_SIZEY-1))
#define SZ ((GLint)S1*(MAX_SIZEZ-1))
#define HALFX (((GLfloat)MAX_SIZEX-1.0)/2.0)
#define HALFY (((GLfloat)MAX_SIZEY-1.0)/2.0)
#define HALFZ (((GLfloat)MAX_SIZEZ-1.0)/2.0)
#define MIDX(a) (((GLfloat)(2*a-MAX_SIZEX+1))/2.0)
#define MIDY(a) (((GLfloat)(2*a-MAX_SIZEY+1))/2.0)
#define MIDZ(a) (((GLfloat)(2*a-MAX_SIZEZ+1))/2.0)
#define DRAW_CUBIT(w,b,f,l,r,bm,t) if (!drawCubit(w,b,f,l,r,bm,t)) return False
	RubikSlice slice = {NO_FACE, NO_ROTATION, NO_DEPTH};
	GLfloat rotateStep = 0.0;
	int i, j, k;

	if (w->rubikGL.movement.face == NO_FACE) {
		slice.face = NO_FACE;
		slice.rotation = NO_ROTATION;
		slice.depth = NO_DEPTH;
	} else if (w->rubikGL.movement.face >= 0 &&
			w->rubikGL.movement.face < MAX_FACES) {
		convertMove(w, w->rubikGL.movement, &slice);
	}
	if (slice.depth != NO_DEPTH)
		rotateStep = (slice.rotation == CCW) ? w->rubikGL.rotateStep :
			-w->rubikGL.rotateStep;
/*-
 * The glRotatef() routine transforms the coordinate system for every future
 * vertex specification (this is not so simple, but by now comprehending this
 * is sufficient). So if you want to rotate the inner slice, you can draw
 * one slice, rotate the anglestep for the centerslice, draw the inner slice,
 * rotate reversely and draw the other slice.
 * There is a sequence for drawing cubies for each axis being moved...
 */
#if 0
slice.face = 0;
slice.face = FRONT_FACE;
(void) printf("slice.face %d\n", slice.face);
#endif
	switch (slice.face) {
	case NO_FACE:
	case TOP_FACE:	/* BOTTOM_FACE too */
		glPushMatrix();
		if (MAX_SIZEY > 1 && slice.depth == MAX_SIZEY - 1)
			glRotatef(rotateStep, 0, HALFY, 0);

		glTranslatef(-HALFX, -HALFY, -HALFZ);
		DRAW_CUBIT(w,
			FIRSTX + MAX_SIZEX * FIRSTY, (MAX_SIZEZ == 1) ?
			FIRSTX + MAX_SIZEX * LASTY : NO_POSITION,
			FIRSTZ + MAX_SIZEZ * LASTY, (MAX_SIZEX == 1) ?
			LASTZ + MAX_SIZEZ * LASTY : NO_POSITION,
			FIRSTX + MAX_SIZEX * LASTZ, (MAX_SIZEY == 1) ?
			FIRSTX + MAX_SIZEX * FIRSTZ : NO_POSITION);
		for (k = 1; k < MAX_SIZEZ - 1; k++) {
			glTranslatef(0, 0, S1);
			DRAW_CUBIT(w,
				NO_POSITION, NO_POSITION,
				k + MAX_SIZEZ * LASTY, (MAX_SIZEX == 1) ?
				REVZ(k) + MAX_SIZEZ * LASTY : NO_POSITION,
				FIRSTX + MAX_SIZEX * REVZ(k), (MAX_SIZEY == 1) ?
				FIRSTX + MAX_SIZEX * k : NO_POSITION);
		}
		if (MAX_SIZEZ > 1) {
			glTranslatef(0, 0, S1);
			DRAW_CUBIT(w,
				NO_POSITION, FIRSTX + MAX_SIZEX * LASTY,
				LASTZ + MAX_SIZEZ * LASTY, (MAX_SIZEX == 1) ?
				FIRSTZ + MAX_SIZEZ * LASTY : NO_POSITION,
				FIRSTX + MAX_SIZEX * FIRSTZ, (MAX_SIZEY == 1) ?
				FIRSTX + MAX_SIZEX * LASTZ : NO_POSITION);
			}
			for (i = 1; i < MAX_SIZEX - 1; i++) {
				glTranslatef(S1, 0, -SZ);
				DRAW_CUBIT(w,
					i + MAX_SIZEX * FIRSTY, (MAX_SIZEZ == 1) ?
					i + MAX_SIZEX * LASTY : NO_POSITION,
					NO_POSITION, NO_POSITION,
					i + MAX_SIZEX * LASTZ, (MAX_SIZEY == 1) ?
					i + MAX_SIZEX * FIRSTZ : NO_POSITION);
				for (k = 1; k < MAX_SIZEZ - 1; k++) {
					glTranslatef(0, 0, S1);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						NO_POSITION, NO_POSITION,
						i + MAX_SIZEX * REVZ(k), (MAX_SIZEY == 1) ?
						i + MAX_SIZEX * k : NO_POSITION);
				}
				if (MAX_SIZEZ > 1) {
					glTranslatef(0, 0, S1);
					DRAW_CUBIT(w,
						NO_POSITION, i + MAX_SIZEX * LASTY,
						NO_POSITION, NO_POSITION,
						i + MAX_SIZEX * FIRSTZ, (MAX_SIZEY == 1) ?
						i + MAX_SIZEX * LASTZ : NO_POSITION);
				}
			}
			if (MAX_SIZEX > 1) {
			glTranslatef(S1, 0, -SZ);
			DRAW_CUBIT(w,
				LASTX + MAX_SIZEX * FIRSTY, (MAX_SIZEZ == 1) ?
				LASTX + MAX_SIZEX * LASTY : NO_POSITION,
				NO_POSITION, LASTZ + MAX_SIZEZ * LASTY,
				LASTX + MAX_SIZEX * LASTZ, (MAX_SIZEY == 1) ?
				LASTX + MAX_SIZEX * FIRSTZ : NO_POSITION);
			if (MAX_SIZEZ > 1) {
				for (k = 1; k < MAX_SIZEZ - 1; k++) {
					glTranslatef(0, 0, S1);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						NO_POSITION, REVZ(k) + MAX_SIZEZ * LASTY,
						LASTX + MAX_SIZEX * REVZ(k), (MAX_SIZEY == 1) ?
						LASTX + MAX_SIZEX * k : NO_POSITION);
				}
			 	glTranslatef(0, 0, S1);
				DRAW_CUBIT(w,
					NO_POSITION, LASTX + MAX_SIZEX * LASTY,
					NO_POSITION, FIRSTZ + MAX_SIZEZ * LASTY,
					LASTX + MAX_SIZEX * FIRSTZ, (MAX_SIZEY == 1) ?
					LASTX + MAX_SIZEX * LASTZ : NO_POSITION);
				}
			}
			glPopMatrix();
			for (j = 1; j < MAX_SIZEY - 1; j++) {
				glPushMatrix();
				if (slice.depth == REVY(j))
					glRotatef(rotateStep, 0, HALFY, 0);
				glTranslatef(-HALFX, MIDY(j), -HALFZ);
				DRAW_CUBIT(w,
					FIRSTX + MAX_SIZEX * j, (MAX_SIZEZ == 1) ?
					FIRSTX + MAX_SIZEX * REVY(j) : NO_POSITION,
					FIRSTZ + MAX_SIZEZ * REVY(j), (MAX_SIZEX == 1) ?
					LASTZ + MAX_SIZEZ * REVY(j) : NO_POSITION,
					NO_POSITION, NO_POSITION);
				for (k = 1; k < MAX_SIZEZ - 1; k++) {
					glTranslatef(0, 0, S1);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						k + MAX_SIZEZ * REVY(j), (MAX_SIZEX == 1) ?
						REVZ(k) + MAX_SIZEZ * REVY(j) : NO_POSITION,
						NO_POSITION, NO_POSITION);
				}
				if (MAX_SIZEZ > 1) {
					glTranslatef(0, 0, S1);
					DRAW_CUBIT(w,
						NO_POSITION, FIRSTX + MAX_SIZEX * REVY(j),
						LASTZ + MAX_SIZEZ * REVY(j), (MAX_SIZEX == 1) ?
						FIRSTZ + MAX_SIZEZ * REVY(j) : NO_POSITION,
						NO_POSITION, NO_POSITION);
				}
				for (i = 1; i < MAX_SIZEX - 1; i++) {
					glTranslatef(1, 0, -SZ);
					DRAW_CUBIT(w,
						i + MAX_SIZEX * j, (MAX_SIZEZ == 1) ?
						i + MAX_SIZEX * REVY(j) : NO_POSITION,
						NO_POSITION, NO_POSITION,
						NO_POSITION, NO_POSITION);
					/* Center */
					if (MAX_SIZEZ > 1) {
					glTranslatef(0, 0, SZ);
					DRAW_CUBIT(w,
						NO_POSITION, i + MAX_SIZEX * REVY(j),
						NO_POSITION, NO_POSITION,
						NO_POSITION, NO_POSITION);
					}
				}
				if (MAX_SIZEX > 1) {
					glTranslatef(S1, 0, -SZ);
					DRAW_CUBIT(w,
						LASTX + MAX_SIZEX * j, (MAX_SIZEZ == 1) ?
						LASTX + MAX_SIZEX * REVY(j) : NO_POSITION,
						NO_POSITION, LASTZ + MAX_SIZEZ * REVY(j),
						NO_POSITION, NO_POSITION);
					for (k = 1; k < MAX_SIZEZ - 1; k++) {
						glTranslatef(0, 0, S1);
						DRAW_CUBIT(w,
							NO_POSITION, NO_POSITION,
							NO_POSITION, REVZ(k) + MAX_SIZEZ * REVY(j),
							NO_POSITION, NO_POSITION);
					}
					if (MAX_SIZEZ > 1) {
						glTranslatef(0, 0, S1);
						DRAW_CUBIT(w,
							NO_POSITION, LASTX + MAX_SIZEX * REVY(j),
							NO_POSITION, FIRSTZ + MAX_SIZEZ * REVY(j),
							NO_POSITION, NO_POSITION);
					}
				}
				glPopMatrix();
			}
			if (MAX_SIZEY > 1) {
				if (slice.depth == 0)
					glRotatef(rotateStep, 0, HALFY, 0);
			glTranslatef(-HALFX, HALFY, -HALFZ);
			DRAW_CUBIT(w,
				FIRSTX + MAX_SIZEX * LASTY, (MAX_SIZEZ == 1) ?
				FIRSTX + MAX_SIZEX * FIRSTY : NO_POSITION,
				FIRSTZ + MAX_SIZEZ * FIRSTY, (MAX_SIZEX == 1) ?
				LASTZ + MAX_SIZEZ * FIRSTY : NO_POSITION,
				NO_POSITION, FIRSTX + MAX_SIZEX * FIRSTZ);
			for (k = 1; k < MAX_SIZEZ - 1; k++) {
				glTranslatef(0, 0, S1);
				DRAW_CUBIT(w,
					NO_POSITION, NO_POSITION,
					k + MAX_SIZEZ * FIRSTY, (MAX_SIZEX == 1) ?
					REVZ(k) + MAX_SIZEZ * FIRSTY : NO_POSITION,
					NO_POSITION, FIRSTX + MAX_SIZEX * k);
			}
			if (MAX_SIZEZ > 1) {
				glTranslatef(0, 0, S1);
				DRAW_CUBIT(w,
					NO_POSITION, FIRSTX + MAX_SIZEX * FIRSTY,
					LASTZ + MAX_SIZEZ * FIRSTY, (MAX_SIZEX == 1) ?
					FIRSTZ + MAX_SIZEZ * FIRSTY : NO_POSITION,
					NO_POSITION, FIRSTX + MAX_SIZEX * LASTZ);
			}
			for (i = 1; i < MAX_SIZEX - 1; i++) {
				glTranslatef(S1, 0, -SZ);
				DRAW_CUBIT(w,
					i + MAX_SIZEX * LASTY, (MAX_SIZEZ == 1) ?
					i + MAX_SIZEX * FIRSTY : NO_POSITION,
					NO_POSITION, NO_POSITION,
					NO_POSITION, i + MAX_SIZEX * FIRSTZ);
				for (k = 1; k < MAX_SIZEZ - 1; k++) {
					glTranslatef(0, 0, S1);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						NO_POSITION, NO_POSITION,
						NO_POSITION, i + MAX_SIZEX * k);
				}
				if (MAX_SIZEZ > 1) {
					glTranslatef(0, 0, S1);
					DRAW_CUBIT(w,
						NO_POSITION, i + MAX_SIZEX * FIRSTY,
						NO_POSITION, NO_POSITION,
						NO_POSITION, i + MAX_SIZEX * LASTZ);
				}
			}
			if (MAX_SIZEX > 1) {
				glTranslatef(S1, 0, -SZ);
				DRAW_CUBIT(w,
					LASTX + MAX_SIZEX * LASTY, (MAX_SIZEZ == 1) ?
					LASTX + MAX_SIZEX * FIRSTY : NO_POSITION,
					NO_POSITION, LASTZ + MAX_SIZEZ * FIRSTY,
					NO_POSITION, LASTX + MAX_SIZEX * FIRSTZ);
				for (k = 1; k < MAX_SIZEZ - 1; k++) {
					glTranslatef(0, 0, S1);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						NO_POSITION, REVZ(k) + MAX_SIZEZ * FIRSTY,
						NO_POSITION, LASTX + MAX_SIZEX * k);
				}
				if (MAX_SIZEZ > 1) {
					glTranslatef(0, 0, S1);
					DRAW_CUBIT(w,
						NO_POSITION, LASTX + MAX_SIZEX * FIRSTY,
						NO_POSITION, FIRSTZ + MAX_SIZEZ * FIRSTY,
						NO_POSITION, LASTX + MAX_SIZEX * LASTZ);
				}
			}
		}
		break;
	case LEFT_FACE:	/* RIGHT_FACE too */
		/* rotateStep is negative because the RIGHT face is the default here */
		glPushMatrix();
		if (MAX_SIZEX > 1 && slice.depth == 0)
			glRotatef(-rotateStep, HALFX, 0, 0);

		glTranslatef(-HALFX, -HALFY, -HALFZ);
		DRAW_CUBIT(w,
			FIRSTX + MAX_SIZEX * FIRSTY, (MAX_SIZEZ == 1) ?
			FIRSTX + MAX_SIZEX * LASTY : NO_POSITION,
			FIRSTZ + MAX_SIZEZ * LASTY, (MAX_SIZEX == 1) ?
			LASTZ + MAX_SIZEZ * LASTY : NO_POSITION,
			FIRSTX + MAX_SIZEX * LASTZ, (MAX_SIZEY == 1) ?
			FIRSTX + MAX_SIZEX * FIRSTZ : NO_POSITION);
		for (j = 1; j < MAX_SIZEY - 1; j++) {
			glTranslatef(0, S1, 0);
			DRAW_CUBIT(w,
				FIRSTX + MAX_SIZEX * j, (MAX_SIZEZ == 1) ?
				FIRSTX + MAX_SIZEX * REVY(j) : NO_POSITION,
				FIRSTZ + MAX_SIZEZ * REVY(j), (MAX_SIZEX == 1) ?
				LASTZ + MAX_SIZEZ * REVY(j) : NO_POSITION,
				NO_POSITION, NO_POSITION);
		}
		if (MAX_SIZEY > 1) {
			glTranslatef(0, S1, 0);
			DRAW_CUBIT(w,
				FIRSTX + MAX_SIZEX * LASTY, (MAX_SIZEZ == 1) ?
				FIRSTX + MAX_SIZEX * FIRSTY : NO_POSITION,
				FIRSTZ + MAX_SIZEZ * FIRSTY, (MAX_SIZEX == 1) ?
				LASTZ + MAX_SIZEZ * FIRSTY : NO_POSITION,
				NO_POSITION, FIRSTX + MAX_SIZEX * FIRSTZ);
		}
		for (k = 1; k < MAX_SIZEZ - 1; k++) {
			glTranslatef(0, -SY, S1);
			DRAW_CUBIT(w,
				NO_POSITION, NO_POSITION,
				k + MAX_SIZEZ * LASTY, (MAX_SIZEX == 1) ?
				REVZ(k) + MAX_SIZEZ * LASTY : NO_POSITION,
				FIRSTX + MAX_SIZEX * REVZ(k), (MAX_SIZEY == 1) ?
				FIRSTX + MAX_SIZEX * k : NO_POSITION);
			for (j = 1; j < MAX_SIZEY - 1; j++) {
				glTranslatef(0, S1, 0);
				DRAW_CUBIT(w,
					NO_POSITION, NO_POSITION,
					k + MAX_SIZEZ * REVY(j), (MAX_SIZEX == 1) ?
					REVZ(k) + MAX_SIZEZ * REVY(j) : NO_POSITION,
					NO_POSITION, NO_POSITION);
			}
			if (MAX_SIZEY > 1) {
				glTranslatef(0, S1, 0);
				DRAW_CUBIT(w,
					NO_POSITION, NO_POSITION,
					k + MAX_SIZEZ * FIRSTY, (MAX_SIZEX == 1) ?
					REVZ(k) + MAX_SIZEZ * FIRSTY : NO_POSITION,
					NO_POSITION, FIRSTX + MAX_SIZEX * k);
			}
		}
		if (MAX_SIZEZ > 1) {
			glTranslatef(0, -SY, S1);
			DRAW_CUBIT(w,
				NO_POSITION, FIRSTX + MAX_SIZEX * LASTY,
				LASTZ + MAX_SIZEZ * LASTY, (MAX_SIZEX == 1) ?
				FIRSTZ + MAX_SIZEZ * LASTY : NO_POSITION,
				FIRSTX + MAX_SIZEX * FIRSTZ, (MAX_SIZEY == 1) ?
				FIRSTX + MAX_SIZEX * LASTZ : NO_POSITION);
			if (MAX_SIZEY > 1) {
				for (j = 1; j < MAX_SIZEY - 1; j++) {
					glTranslatef(0, S1, 0);
					DRAW_CUBIT(w,
						NO_POSITION, FIRSTX + MAX_SIZEX * REVY(j),
						LASTZ + MAX_SIZEZ * REVY(j), (MAX_SIZEX == 1) ?
						FIRSTZ + MAX_SIZEZ * REVY(j) : NO_POSITION,
						NO_POSITION, NO_POSITION);
				}
				glTranslatef(0, S1, 0);
				DRAW_CUBIT(w,
					NO_POSITION, FIRSTX + MAX_SIZEX * FIRSTY,
					LASTZ + MAX_SIZEZ * FIRSTY, (MAX_SIZEX == 1) ?
					FIRSTZ + MAX_SIZEZ * FIRSTY : NO_POSITION,
					NO_POSITION, FIRSTX + MAX_SIZEX * LASTZ);
			}
		}
		glPopMatrix();
		for (i = 1; i < MAX_SIZEX - 1; i++) {
			glPushMatrix();
			if (slice.depth == i)
				glRotatef(-rotateStep, HALFX, 0, 0);
			glTranslatef(MIDX(i), -HALFY, -HALFZ);
			DRAW_CUBIT(w,
				i + MAX_SIZEX * FIRSTY, (MAX_SIZEZ == 1) ?
				i + MAX_SIZEX * LASTY : NO_POSITION,
				NO_POSITION, NO_POSITION,
				i + MAX_SIZEX * LASTZ, (MAX_SIZEY == 1) ?
				i + MAX_SIZEX * FIRSTZ : NO_POSITION);
			for (j = 1; j < MAX_SIZEY - 1; j++) {
				glTranslatef(0, S1, 0);
				DRAW_CUBIT(w,
					i + MAX_SIZEX * j, (MAX_SIZEZ == 1) ?
					i + MAX_SIZEX * REVY(j) : NO_POSITION,
					NO_POSITION, NO_POSITION,
					NO_POSITION, NO_POSITION);
			}
			if (MAX_SIZEY > 1) {
				glTranslatef(0, S1, 0);
				DRAW_CUBIT(w,
					i + MAX_SIZEX * LASTY, (MAX_SIZEZ == 1) ?
					i + MAX_SIZEX * FIRSTY : NO_POSITION,
					NO_POSITION, NO_POSITION,
					NO_POSITION, i + MAX_SIZEX * FIRSTZ);
			}
			for (k = 1; k < MAX_SIZEZ - 1; k++) {
				glTranslatef(0, -SY, S1);
				DRAW_CUBIT(w,
					NO_POSITION, NO_POSITION,
					NO_POSITION, NO_POSITION,
					i + MAX_SIZEX * REVZ(k), (MAX_SIZEY == 1) ?
					i + MAX_SIZEX * k : NO_POSITION);
				/* Center */
				if (MAX_SIZEY > 1) {
					glTranslatef(0, SY, 0);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						NO_POSITION, NO_POSITION,
						NO_POSITION, i + MAX_SIZEX * k);
				}
			}
			if (MAX_SIZEZ > 1) {
				glTranslatef(0, -SY, S1);
				DRAW_CUBIT(w,
					NO_POSITION, i + MAX_SIZEX * LASTY,
					NO_POSITION, NO_POSITION,
					i + MAX_SIZEX * FIRSTZ, (MAX_SIZEY == 1) ?
					i + MAX_SIZEX * LASTZ : NO_POSITION);
				for (j = 1; j < MAX_SIZEY - 1; j++) {
					glTranslatef(0, S1, 0);
					DRAW_CUBIT(w,
						NO_POSITION, i + MAX_SIZEX * REVY(j),
						NO_POSITION, NO_POSITION,
						NO_POSITION, NO_POSITION);
				}
				if (MAX_SIZEY > 1) {
					glTranslatef(0, S1, 0);
					DRAW_CUBIT(w,
						NO_POSITION, i + MAX_SIZEX * FIRSTY,
						NO_POSITION, NO_POSITION,
						NO_POSITION, i + MAX_SIZEX * LASTZ);
				}
			}
			glPopMatrix();
		}
		if (MAX_SIZEX > 1) {
			if (slice.depth == MAX_SIZEX - 1)
				glRotatef(-rotateStep, HALFX, 0, 0);
			glTranslatef(HALFX, -HALFY, -HALFZ);
			DRAW_CUBIT(w,
				LASTX + MAX_SIZEX * FIRSTY, (MAX_SIZEZ == 1) ?
				LASTX + MAX_SIZEX * LASTY : NO_POSITION,
				NO_POSITION, LASTZ + MAX_SIZEZ * LASTY,
				LASTX + MAX_SIZEX * LASTZ, (MAX_SIZEY == 1) ?
				LASTX + MAX_SIZEX * FIRSTZ : NO_POSITION);
			for (j = 1; j < MAX_SIZEY - 1; j++) {
				glTranslatef(0, S1, 0);
				DRAW_CUBIT(w,
					LASTX + MAX_SIZEX * j, (MAX_SIZEZ == 1) ?
					LASTX + MAX_SIZEX * REVY(j) : NO_POSITION,
					NO_POSITION, LASTZ + MAX_SIZEZ * REVY(j),
					NO_POSITION, NO_POSITION);
			}
			if (MAX_SIZEY > 1) {
				glTranslatef(0, S1, 0);
				DRAW_CUBIT(w,
					LASTX + MAX_SIZEX * LASTY, (MAX_SIZEZ == 1) ?
					LASTX + MAX_SIZEX * FIRSTY : NO_POSITION,
					NO_POSITION, LASTZ + MAX_SIZEZ * FIRSTY,
					NO_POSITION, LASTX + MAX_SIZEX * FIRSTZ);
			}
			for (k = 1; k < MAX_SIZEZ - 1; k++) {
				glTranslatef(0, -SY, S1);
				DRAW_CUBIT(w,
					NO_POSITION, NO_POSITION,
					NO_POSITION, REVZ(k) + MAX_SIZEZ * LASTY,
					LASTX + MAX_SIZEX * REVZ(k), (MAX_SIZEY == 1) ?
					LASTX + MAX_SIZEX * k : NO_POSITION);
				for (j = 1; j < MAX_SIZEY - 1; j++) {
					glTranslatef(0, S1, 0);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						NO_POSITION, REVZ(k) + MAX_SIZEZ * REVY(j),
						NO_POSITION, NO_POSITION);
				}
				if (MAX_SIZEY > 1) {
					glTranslatef(0, S1, 0);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						NO_POSITION, REVZ(k) + MAX_SIZEZ * FIRSTY,
						NO_POSITION, LASTX + MAX_SIZEX * k);
				}
			}
			if (MAX_SIZEZ > 1) {
				glTranslatef(0, -SY, S1);
				DRAW_CUBIT(w,
					NO_POSITION, LASTX + MAX_SIZEX * LASTY,
					NO_POSITION, FIRSTZ + MAX_SIZEZ * LASTY,
					LASTX + MAX_SIZEX * FIRSTZ, (MAX_SIZEY == 1) ?
					LASTX + MAX_SIZEX * LASTZ : NO_POSITION);
				for (j = 1; j < MAX_SIZEY - 1; j++) {
					glTranslatef(0, S1, 0);
					DRAW_CUBIT(w,
						NO_POSITION, LASTX + MAX_SIZEX * REVY(j),
						NO_POSITION, FIRSTZ + MAX_SIZEZ * REVY(j),
						NO_POSITION, NO_POSITION);
				}
				if (MAX_SIZEY > 1) {
					glTranslatef(0, S1, 0);
					DRAW_CUBIT(w,
						NO_POSITION, LASTX + MAX_SIZEX * FIRSTY,
						NO_POSITION, FIRSTZ + MAX_SIZEZ * FIRSTY,
						NO_POSITION, LASTX + MAX_SIZEX * LASTZ);
				}
			}
		}
		break;
	case FRONT_FACE:	/* BACK_FACE too */
		glPushMatrix();
		if (MAX_SIZEZ > 1 && slice.depth == MAX_SIZEZ - 1)
			glRotatef(rotateStep, 0, 0, HALFZ);

		glTranslatef(-HALFX, -HALFY, -HALFZ);
		DRAW_CUBIT(w,
			FIRSTX + MAX_SIZEX * FIRSTY, (MAX_SIZEZ == 1) ?
			FIRSTX + MAX_SIZEX * LASTY : NO_POSITION,
			FIRSTZ + MAX_SIZEZ * LASTY, (MAX_SIZEX == 1) ?
			LASTZ + MAX_SIZEZ * LASTY : NO_POSITION,
			FIRSTX + MAX_SIZEX * LASTZ, (MAX_SIZEY == 1) ?
			FIRSTX + MAX_SIZEX * FIRSTZ : NO_POSITION);
		for (i = 1; i < MAX_SIZEX - 1; i++) {
			glTranslatef(S1, 0, 0);
			DRAW_CUBIT(w,
				i + MAX_SIZEX * FIRSTY, (MAX_SIZEZ == 1) ?
				i + MAX_SIZEX * LASTY : NO_POSITION,
				NO_POSITION, NO_POSITION,
				i + MAX_SIZEX * LASTZ, (MAX_SIZEY == 1) ?
				i + MAX_SIZEX * FIRSTZ : NO_POSITION);
		}
		if (MAX_SIZEX > 1) {
			glTranslatef(S1, 0, 0);
			DRAW_CUBIT(w,
				LASTX + MAX_SIZEX * FIRSTY, (MAX_SIZEZ == 1) ?
				LASTX + MAX_SIZEX * LASTY : NO_POSITION,
				NO_POSITION, LASTZ + MAX_SIZEZ * LASTY,
				LASTX + MAX_SIZEX * LASTZ, (MAX_SIZEY == 1) ?
				LASTX + MAX_SIZEX * FIRSTZ : NO_POSITION);
		}
		for (j = 1; j < MAX_SIZEY - 1; j++) {
			glTranslatef(-SX, S1, 0);
			DRAW_CUBIT(w,
				FIRSTX + MAX_SIZEX * j, (MAX_SIZEZ == 1) ?
				FIRSTX + MAX_SIZEX * REVY(j) : NO_POSITION,
				FIRSTZ + MAX_SIZEZ * REVY(j), (MAX_SIZEX == 1) ?
				LASTZ + MAX_SIZEZ * REVY(j) : NO_POSITION,
				NO_POSITION, NO_POSITION);
			for (i = 1; i < MAX_SIZEX - 1; i++) {
				glTranslatef(S1, 0, 0);
				DRAW_CUBIT(w,
					i + MAX_SIZEX * j, (MAX_SIZEZ == 1) ?
					i + MAX_SIZEX * REVY(j) : NO_POSITION,
					NO_POSITION, NO_POSITION,
					NO_POSITION, NO_POSITION);
			}
			if (MAX_SIZEX > 1) {
				glTranslatef(S1, 0, 0);
				DRAW_CUBIT(w,
					LASTX + MAX_SIZEX * j, (MAX_SIZEZ == 1) ?
					LASTX + MAX_SIZEX * REVY(j) : NO_POSITION,
					NO_POSITION, LASTZ + MAX_SIZEZ * REVY(j),
					NO_POSITION, NO_POSITION);
			}
		}
		if (MAX_SIZEY > 1) {
			glTranslatef(-SX, S1, 0);
			DRAW_CUBIT(w,
				FIRSTX + MAX_SIZEX * LASTY, (MAX_SIZEZ == 1) ?
				FIRSTX + MAX_SIZEX * FIRSTY : NO_POSITION,
				FIRSTZ + MAX_SIZEZ * FIRSTY, (MAX_SIZEX == 1) ?
				LASTZ + MAX_SIZEZ * FIRSTY : NO_POSITION,
				NO_POSITION, FIRSTX + MAX_SIZEX * FIRSTZ);
			if (MAX_SIZEX > 1) {
				for (i = 1; i < MAX_SIZEX - 1; i++) {
					glTranslatef(S1, 0, 0);
					DRAW_CUBIT(w,
						i + MAX_SIZEX * LASTY, (MAX_SIZEZ == 1) ?
						i + MAX_SIZEX * FIRSTY : NO_POSITION,
						NO_POSITION, NO_POSITION,
						NO_POSITION, i + MAX_SIZEX * FIRSTZ);
				}
				glTranslatef(S1, 0, 0);
				DRAW_CUBIT(w,
					LASTX + MAX_SIZEX * LASTY, (MAX_SIZEZ == 1) ?
					LASTX + MAX_SIZEX * FIRSTY : NO_POSITION,
					NO_POSITION, LASTZ + MAX_SIZEZ * FIRSTY,
					NO_POSITION, LASTX + MAX_SIZEX * FIRSTZ);
			}
		}
		glPopMatrix();
		for (k = 1; k < MAX_SIZEZ - 1; k++) {
			glPushMatrix();
			if (slice.depth == REVZ(k))
				glRotatef(rotateStep, 0, 0, HALFZ);
			glTranslatef(-HALFX, -HALFY, MIDZ(k));
			DRAW_CUBIT(w,
				NO_POSITION, NO_POSITION,
				k + MAX_SIZEZ * LASTY, (MAX_SIZEX == 1) ?
				REVZ(k) + MAX_SIZEZ * LASTY : NO_POSITION,
				FIRSTX + MAX_SIZEX * REVZ(k), (MAX_SIZEY == 1) ?
				FIRSTX + MAX_SIZEX * k : NO_POSITION);
			for (i = 1; i < MAX_SIZEX - 1; i++) {
				glTranslatef(S1, 0, 0);
				DRAW_CUBIT(w,
					NO_POSITION, NO_POSITION,
					NO_POSITION, NO_POSITION,
					i + MAX_SIZEX * REVZ(k), (MAX_SIZEY == 1) ?
					i + MAX_SIZEX * k : NO_POSITION);
			}
			if (MAX_SIZEX > 1) {
				glTranslatef(S1, 0, 0);
				DRAW_CUBIT(w,
					NO_POSITION, NO_POSITION,
					NO_POSITION, REVZ(k) + MAX_SIZEZ * LASTY,
					LASTX + MAX_SIZEX * REVZ(k), (MAX_SIZEY == 1) ?
					LASTX + MAX_SIZEX * k : NO_POSITION);
			}
			for (j = 1; j < MAX_SIZEY - 1; j++) {
				glTranslatef(-SX, S1, 0);
				DRAW_CUBIT(w,
					NO_POSITION, NO_POSITION,
					k + MAX_SIZEZ * REVY(j), (MAX_SIZEX == 1) ?
					REVZ(k) + MAX_SIZEZ * REVY(j) : NO_POSITION,
					NO_POSITION, NO_POSITION);
				/* Center */
				if (MAX_SIZEX > 1) {
					glTranslatef(SX, 0, 0);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						NO_POSITION, REVZ(k) + MAX_SIZEZ * REVY(j),
						NO_POSITION, NO_POSITION);
				}
			}
			if (MAX_SIZEY > 1) {
				glTranslatef(-SX, S1, 0);
				DRAW_CUBIT(w,
					NO_POSITION, NO_POSITION,
					k + MAX_SIZEZ * FIRSTY, (MAX_SIZEX == 1) ?
					REVZ(k) + MAX_SIZEZ * FIRSTY : NO_POSITION,
					NO_POSITION, FIRSTX + MAX_SIZEX * k);
				for (i = 1; i < MAX_SIZEX - 1; i++) {
					glTranslatef(S1, 0, 0);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						NO_POSITION, NO_POSITION,
						NO_POSITION, i + MAX_SIZEX * k);
				}
				if (MAX_SIZEX > 1) {
					glTranslatef(S1, 0, 0);
					DRAW_CUBIT(w,
						NO_POSITION, NO_POSITION,
						NO_POSITION, REVZ(k) + MAX_SIZEZ * FIRSTY,
						NO_POSITION, LASTX + MAX_SIZEX * k);
				}
			}
			glPopMatrix();
		}
		if (MAX_SIZEZ > 1) {
			if (slice.depth == 0)
				glRotatef(rotateStep, 0, 0, HALFZ);
			glTranslatef(-HALFX, -HALFY, HALFZ);
			DRAW_CUBIT(w,
				NO_POSITION, FIRSTX + MAX_SIZEX * LASTY,
				LASTZ + MAX_SIZEZ * LASTY, (MAX_SIZEX == 1) ?
				FIRSTZ + MAX_SIZEZ * LASTY : NO_POSITION,
				FIRSTX + MAX_SIZEX * FIRSTZ, (MAX_SIZEY == 1) ?
				FIRSTX + MAX_SIZEX * LASTZ : NO_POSITION);
			for (i = 1; i < MAX_SIZEX - 1; i++) {
				glTranslatef(S1, 0, 0);
				DRAW_CUBIT(w,
					NO_POSITION, i + MAX_SIZEX * LASTY,
					NO_POSITION, NO_POSITION,
					i + MAX_SIZEX * FIRSTZ, (MAX_SIZEY == 1) ?
					i + MAX_SIZEX * LASTZ : NO_POSITION);
			}
			if (MAX_SIZEX > 1) {
				glTranslatef(S1, 0, 0);
				DRAW_CUBIT(w,
					NO_POSITION, LASTX + MAX_SIZEX * LASTY,
					NO_POSITION, FIRSTZ + MAX_SIZEZ * LASTY,
					LASTX + MAX_SIZEX * FIRSTZ, (MAX_SIZEY == 1) ?
					LASTX + MAX_SIZEX * LASTZ : NO_POSITION);
			}
			for (j = 1; j < MAX_SIZEY - 1; j++) {
				glTranslatef(-SX, S1, 0);
				DRAW_CUBIT(w,
					NO_POSITION, FIRSTX + MAX_SIZEX * REVY(j),
					LASTZ + MAX_SIZEZ * REVY(j), (MAX_SIZEX == 1) ?
					FIRSTZ + MAX_SIZEZ * REVY(j) : NO_POSITION,
					NO_POSITION, NO_POSITION);
				for (i = 1; i < MAX_SIZEX - 1; i++) {
					glTranslatef(S1, 0, 0);
					DRAW_CUBIT(w,
						NO_POSITION, i + MAX_SIZEX * REVY(j),
						NO_POSITION, NO_POSITION,
						NO_POSITION, NO_POSITION);
				}
				if (MAX_SIZEX > 1) {
					glTranslatef(S1, 0, 0);
					DRAW_CUBIT(w,
						NO_POSITION, LASTX + MAX_SIZEX * REVY(j),
						NO_POSITION, FIRSTZ + MAX_SIZEZ * REVY(j),
						NO_POSITION, NO_POSITION);
				}
			}
			if (MAX_SIZEY > 1) {
				glTranslatef(-SX, S1, 0);
				DRAW_CUBIT(w,
					NO_POSITION, FIRSTX + MAX_SIZEX * FIRSTY,
					LASTZ + MAX_SIZEZ * FIRSTY, (MAX_SIZEX == 1) ?
					FIRSTZ + MAX_SIZEZ * FIRSTY : NO_POSITION,
					NO_POSITION, FIRSTX + MAX_SIZEX * LASTZ);
				for (i = 1; i < MAX_SIZEX - 1; i++) {
					glTranslatef(S1, 0, 0);
					DRAW_CUBIT(w,
						NO_POSITION, i + MAX_SIZEX * FIRSTY,
						NO_POSITION, NO_POSITION,
						NO_POSITION, i + MAX_SIZEX * LASTZ);
				}
				if (MAX_SIZEX > 1) {
					glTranslatef(S1, 0, 0);
					DRAW_CUBIT(w,
						NO_POSITION, LASTX + MAX_SIZEX * FIRSTY,
						NO_POSITION, FIRSTZ + MAX_SIZEZ * FIRSTY,
						NO_POSITION, LASTX + MAX_SIZEX * LASTZ);
				}
			}
		}
		break;
	}
	return True;
#undef S1
}

void
movePiecesGL(RubikGLWidget w, int face, int position, int direction,
		Boolean control, int fast)
{
	int i, f;
	float length = (w->rubik.degreeTurn == 90) ? 1.0 : 2.0;
	/* Longer so move a little faster */
	float speed = (w->rubik.degreeTurn == 90) ? 1.0 : (2.0 / 3.0);
	double angleStep = 0.0;

	w->rubikGL.movement.face = face;
	w->rubikGL.movement.position = position;
	w->rubikGL.movement.direction = direction;
	w->rubikGL.rotateStep = 0.0;
	w->rubikGL.angleStepX = 0.0;
	w->rubikGL.angleStepY = 0.0;
	w->rubikGL.angleStepZ = 0.0;
	f = (w->rubikGL.numTurnInc < fast) ? 1 : fast;
	for (i = 0; i < w->rubikGL.numTurnInc * length / f; i++) {
		double increment = 90.0 * f / w->rubikGL.numTurnInc;

		if (control) { /* move whole puzzle */
			angleStep += increment;
			if (rotateDir[face][direction] == -1)
				w->rubikGL.angleStepX = -angleStep;
			else if (rotateDir[face][direction] == 1)
				w->rubikGL.angleStepX = angleStep;
			else if (rotateDir[face][direction] == -2)
				w->rubikGL.angleStepY = -angleStep;
			else if (rotateDir[face][direction] == 2)
				w->rubikGL.angleStepY = angleStep;
			else if (rotateDir[face][direction] == -3)
				w->rubikGL.angleStepZ = -angleStep;
			else if (rotateDir[face][direction] == 3)
				w->rubikGL.angleStepZ = angleStep;
		} else {
			w->rubikGL.rotateStep += increment;
		}
		drawAllPiecesGL(w);
		Sleep((unsigned int) ((w->rubik.delay /
			(w->rubikGL.numTurnInc * f)) * speed * 20.0));
		w->rubik.currentFace = -1;
		w->rubik.currentPosition = -1;
	}
	w->rubikGL.angleStepX = 0.0;
	w->rubikGL.angleStepY = 0.0;
	w->rubikGL.angleStepZ = 0.0;
	w->rubikGL.movement.face = NO_FACE;
	w->rubikGL.movement.position = NO_POSITION;
	w->rubikGL.movement.direction = NO_DEPTH;
}

void
drawAllPiecesGL(RubikGLWidget w)
{
#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("Draw All GL error");
	}
#endif
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);
	glPushMatrix();
	glTranslatef(0.0, 0.0, -10.0);
	if (w->core.height > w->core.width)
		glScalef(SCALE, SCALE * w->core.width / w->core.height, SCALE);
	else
		glScalef(SCALE * w->core.height / w->core.width, SCALE, SCALE);
	switch (w->rubik.view) {
	case 0: /* >-  vertical */
		glRotatef(w->rubikGL.step / 2, 0, 1, 0);
		glRotatef(w->rubikGL.step / 2, 1, 0, 0);
		break;
	case 1: /* Y  !vertical */
		glRotatef(w->rubikGL.step / 2, 1, 0, 0);
		glRotatef(w->rubikGL.step / 2, 0, 1, 0);
		break;
	case 2: /* -<  vertical */
		glRotatef(-w->rubikGL.step / 2, 0, 1, 0);
		glRotatef(-3 * w->rubikGL.step / 2, 1, 0, 0);
		break;
	case 3: /* ^  !vertical */
		glRotatef(-w->rubikGL.step / 2, 1, 0, 0);
		glRotatef(-3 * w->rubikGL.step / 2, 0, 1, 0);
		break;
	default:
		break;
	}
	glRotatef(w->rubikGL.angleStepX, 1, 0, 0);
	glRotatef(w->rubikGL.angleStepY, 0, 1, 0);
	glRotatef(w->rubikGL.angleStepZ, 0, 0, 1);
	if (!drawCube(w)) {
		DISPLAY_WARNING("drawCube error");
	}
	glPopMatrix();
	glFlush();
#ifdef WINVER
	SwapBuffers(w->core.hDC);
#else
	glXSwapBuffers(XtDisplay(w), XtWindow(w));
#endif
}

void
drawFrameGL(RubikGLWidget w, Boolean focus)
{
	/* UNUSED */
}

static int
normalizePosition(float x, float y, int sizeX, int sizeY)
{
	float xf, yf;
	int xi, yi;

	xf = (x * sizeX + sizeX) / 2.0;
	yf = (y * sizeY + sizeY) / 2.0;
	xi = (int) xf;
	if (xi < 0)
		xi = 0;
	if (xi >= sizeX)
		xi = sizeX - 1;
	yi = (int) yf;
	if (yi < 0)
		xi = 0;
	if (yi >= sizeY)
		yi = sizeY - 1;
#ifdef DEBUG
	(void) printf("xf %g, yf %g: xi %d, yi %d\n", xf, yf, xi, yi);
#endif
	return xi + yi * sizeX;
}

Boolean
selectPiecesGL(RubikGLWidget w, int x, int y, int *face, int *position)
{
	/* Using gluUnProject By: Luke Benstead */
	/* http://nehe.gamedev.net/data/articles/article.asp?article=13 */

	GLint viewport[4];
	GLdouble modelview[16];
	GLdouble projection[16];
	GLfloat winX, winY, winZ;
	GLdouble posX, posY, posZ;
	Matrix a, b, r;
	Vector t, world, nw;

	glGetDoublev(GL_MODELVIEW_MATRIX, modelview);
	glGetDoublev(GL_PROJECTION_MATRIX, projection);
	glGetIntegerv(GL_VIEWPORT, viewport);

	winX = (float) x;
	winY = (float) viewport[3] - (float) y;
	glReadPixels(x, (int) winY, 1, 1, GL_DEPTH_COMPONENT, GL_FLOAT,
		&winZ);
	gluUnProject(winX, winY, winZ, modelview, projection, viewport,
		&(posX), &(posY), &(posZ));
	t[0] = posX;
	t[1] = posY;
	t[2] = posZ + 10;
	if (w->core.height > w->core.width)
		t[1] *= (double) w->core.height / (double) w->core.width;
	else
		t[0] *= (double) w->core.width / (double) w->core.height;
	*face = -1;
	*position = -1;
	if (winZ == 1) {
		return False;
	}
	world[0] = t[0], world[1] = t[1], world[2] = t[2];
	switch (w->rubik.view) {
	case 0: /* >-  vertical */
		setMatrix(-w->rubikGL.step / 2, 1.0, 0.0, 0.0, a);
		setMatrix(-w->rubikGL.step / 2, 0.0, 1.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / w->rubik.sizex;
		nw[1] = world[1] / w->rubik.sizey;
		nw[2] = world[2] / w->rubik.sizez;
		if (nw[1] >= nw[0] && nw[1] >= nw[2]) {
			*face = 0;
			*position = normalizePosition(world[0], world[2],
				w->rubik.sizex, w->rubik.sizez);
		} else if (nw[0] >= nw[1] && nw[0] >= nw[2]) {
			*face = 1;
			*position = normalizePosition(world[2], -world[1],
				w->rubik.sizez, w->rubik.sizey);
		} else if (nw[2] >= nw[0] && nw[2] >= nw[1]) {
			*face = 2;
			*position = normalizePosition(world[0], -world[1],
				w->rubik.sizex, w->rubik.sizey);
		}
		break;
	case 1: /* Y  !vertical */
		setMatrix(-w->rubikGL.step / 2, 0.0, 1.0, 0.0, a);
		setMatrix(-w->rubikGL.step / 2, 1.0, 0.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / w->rubik.sizex;
		nw[1] = world[1] / w->rubik.sizey;
		nw[2] = world[2] / w->rubik.sizez;
		if (nw[1] >= nw[0] && nw[1] >= nw[2]) {
			*face = 0;
			*position = normalizePosition(world[0], world[2],
				w->rubik.sizex, w->rubik.sizez);
		} else if (nw[0] >= nw[1] && nw[0] >= nw[2]) {
			*face = 1;
			*position = normalizePosition(world[2], -world[1],
				w->rubik.sizez, w->rubik.sizey);
		} else if (nw[2] >= nw[0] && nw[2] >= nw[1]) {
			*face = 2;
			*position = normalizePosition(world[0], -world[1],
				w->rubik.sizex, w->rubik.sizey);
		}
		break;
	case 2: /* -<  vertical */
		setMatrix(3 * w->rubikGL.step / 2, 1.0, 0.0, 0.0, a);
		setMatrix(w->rubikGL.step / 2, 0.0, 1.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / w->rubik.sizex;
		nw[1] = world[1] / w->rubik.sizey;
		nw[2] = world[2] / w->rubik.sizez;
		if (nw[0] <= nw[1] && nw[0] <= nw[2]) {
			*face = 3;
			*position = normalizePosition(-world[2], -world[1],
				w->rubik.sizez, w->rubik.sizey);
		} else if (nw[1] <= nw[0] && nw[1] <= nw[2]) {
			*face = 4;
			*position = normalizePosition(world[0], -world[2],
				w->rubik.sizex, w->rubik.sizez);
		} else if (nw[2] <= nw[0] && nw[2] <= nw[1]) {
			*face = 5;
			*position = normalizePosition(world[0], world[1],
				w->rubik.sizex, w->rubik.sizey);
		}
		break;
	case 3: /* ^  !vertical */
		setMatrix(3 * w->rubikGL.step / 2, 0.0, 1.0, 0.0, a);
		setMatrix(w->rubikGL.step / 2, 1.0, 0.0, 0.0, b);
		matrixMatrix(a, b, r);
		matrixVector(r, t, world);
		nw[0] = -world[0] / w->rubik.sizex;
		nw[1] = world[1] / w->rubik.sizey;
		nw[2] = world[2] / w->rubik.sizez;
		if (nw[0] <= nw[1] && nw[0] <= nw[2]) {
			*face = 3;
			*position = normalizePosition(-world[2], -world[1],
				w->rubik.sizez, w->rubik.sizey);
		} else if (nw[1] <= nw[0] && nw[1] <= nw[2]) {
			*face = 4;
			*position = normalizePosition(world[0], -world[2],
				w->rubik.sizex, w->rubik.sizez);
		} else if (nw[2] <= nw[0] && nw[2] <= nw[1]) {
			*face = 5;
			*position = normalizePosition(world[0], world[1],
				w->rubik.sizex, w->rubik.sizey);
		}
		break;
	default:
		break;
	}
#ifdef DEBUG
	(void) printf("x %d, y %d\n", x, y);
	(void) printf(" wx %g, wy %g, wz %g\n", winX, winY, winZ);
	(void) printf("  px %g, py %g, pz %g\n", posX, posY, posZ);
	(void) printf("   tx %g, ty %g, tz %g\n", t[0], t[1], t[2]);
	(void) printf("    rx %g, ry %g, rz %g\n",
		world[0], world[1], world[2]);
	(void) printf("     face %d, pos %d\n", *face, *position);
#endif
	return True;
}

Boolean
narrowSelectionGL(RubikGLWidget w, int *face, int *position, int *direction)
{
	switch (*direction) {
	case TOP:
	case RIGHT:
	case BOTTOM:
	case LEFT:
		if (w->rubik.view % 2 == 0) {
			if (*face == 1 || *face == 3)
				return False;
		} else {
			if (*face == 0 || *face == 4)
				return False;
			if (*face == 5)
				*direction = (*direction + 2) % MAX_ORIENT;
		}
		break;
	case CCW:
	case CW:
		break;
	case TR:
		if (w->rubik.view % 2 == 0) {
			if (*face == 0 || *face == 5)
				return False;
			else if (*face == 1 || *face == 2 || *face == 4)
				*direction = TOP;
			else	/* (*face == 3) */
				*direction = LEFT;
		} else {
			if (*face == 1 || *face == 5)
				return False;
			else if (*face == 0 || *face == 2 || *face == 3)
				*direction = RIGHT;
			else	/* (*face == 4) */
				*direction = BOTTOM;
		}
		break;
	case BR:
		if (w->rubik.view % 2 == 0) {
			if (*face == 2 || *face == 4)
				return False;
			else if (*face == 0 || *face == 5)
				*direction = BOTTOM;
			else if (*face == 1)
				*direction = RIGHT;
			else	/* (*face == 3) */
				*direction = TOP;
		} else {
			if (*face == 2 || *face == 3)
				return False;
			else if (*face == 4 || *face == 5)
				*direction = LEFT;
			else if (*face == 0)
				*direction = BOTTOM;
			else	/* (*face == 1) */
				*direction = RIGHT;
		}
		break;
	case BL:
		if (w->rubik.view % 2 == 0) {
			if (*face == 0 || *face == 5)
				return False;
			else if (*face == 1 || *face == 2 || *face == 4)
				*direction = BOTTOM;
			else	/* (*face == 3) */
				*direction = RIGHT;
		} else {
			if (*face == 1 || *face == 5)
				return False;
			else if (*face == 0 || *face == 2 || *face == 3)
				*direction = LEFT;
			else	/* (*face == 4) */
				*direction = TOP;
		}
		break;
	case TL:
		if (w->rubik.view % 2 == 0) {
			if (*face == 2 || *face == 4)
				return False;
			else if (*face == 0 || *face == 5)
				*direction = TOP;
			else if (*face == 1)
				*direction = LEFT;
			else	/* (*face == 3) */
				*direction = BOTTOM;
		} else {
			if (*face == 2 || *face == 3)
				return False;
			else if (*face == 4 || *face == 5)
				*direction = RIGHT;
			else if (*face == 0)
				*direction = TOP;
			else	/* (*face == 1) */
				*direction = LEFT;
		}
		break;
	default:
		return False;
	}
	/* Remap to row movement */
	if (*direction == CW || *direction == CCW) {
		int i, j, sizeOfRow, sizeOfColumn, newFace;

		newFace = rotateToRow[*face].face;
		faceSizes((RubikWidget) w, newFace, &sizeOfRow, &sizeOfColumn);
		*direction = (*direction == CCW) ?
			(rotateToRow[*face].direction + 2) % MAX_ORIENT :
			rotateToRow[*face].direction;
		if (rotateToRow[*face].sideFace == LEFT ||
				rotateToRow[*face].sideFace == BOTTOM) {
			i = sizeOfRow - 1;
			j = sizeOfColumn - 1;
		} else {
			i = j = 0;
		}
		*face = newFace;
		*position = j * sizeOfRow + i;
	}
	return True;
}

#ifdef WINVER
static Boolean
setupPixelFormat(RubikGLWidget w, BYTE type, DWORD flags)
{
	PIXELFORMATDESCRIPTOR pfd;
	int pixelFormat;

	memset(&pfd, 0, sizeof(PIXELFORMATDESCRIPTOR));
	pfd.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pfd.nVersion = 1;
	pfd.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | flags;
	pfd.dwLayerMask = PFD_MAIN_PLANE;
	/*pfd.cColorBits = 8;
	pfd.cDepthBits = 16;*/
	pfd.iPixelType = type;
	pfd.cColorBits = 24;
	pfd.cDepthBits = 32;

	if ((pixelFormat = ChoosePixelFormat(w->core.hDC, &pfd)) == 0) {
		DWORD err;
		char *buf1;

		err = GetLastError();
		/* 6 ERROR_INVALID_HANDLE */
		intCat(&buf1, "ChoosePixelFormat failed ", err);
		DISPLAY_WARNING(buf1);
		free(buf1);
		return FALSE;
	}

	if (SetPixelFormat(w->core.hDC, pixelFormat, &pfd) == FALSE) {
		MessageBox(NULL, "SetPixelFormat failed", "Error", MB_OK);
		return FALSE;
	}
	DescribePixelFormat(w->core.hDC, pixelFormat,
		sizeof(PIXELFORMATDESCRIPTOR), &pfd);
	return TRUE;
}

#else

static Boolean
setValuesPuzzleGL(Widget current, Widget request, Widget renew)
{
	RubikGLWidget c = (RubikGLWidget) current, w = (RubikGLWidget) renew;
	Boolean redraw = False;

	if (w->rubik.view != c->rubik.view) {
		resizePuzzleGL(w);
		redraw = True;
	}
	if (w->rubik.sizex != c->rubik.sizex ||
			w->rubik.sizey != c->rubik.sizey ||
			w->rubik.sizez != c->rubik.sizez) {
		resetPieces((RubikWidget) w);
		resizePuzzleGL(w);
		redraw = True;
	}
#if 0
	if (w->rubikGL.cubeSizex.x != c->rubikGL.cubeSizex.x ||
			w->rubikGL.cubeSizey.x != c->rubikGL.cubeSizey.x ||
			w->rubikGL.cubeSizez.x != c->rubikGL.cubeSizez.x ||
			w->rubikGL.cubeSizex.y != c->rubikGL.cubeSizex.y ||
			w->rubikGL.cubeSizey.y != c->rubikGL.cubeSizey.y ||
			w->rubikGL.cubeSizez.y != c->rubikGL.cubeSizez.y) {
		resizePuzzleGL(w);
		redraw = True;
	}
#endif
	if (w->rubik.delay != c->rubik.delay) {
		w->rubikGL.numTurnInc = ((w->rubik.delay < MAX_TURN_INC) ?
			w->rubik.delay + 1 : MAX_TURN_INC);
	}
	return (redraw);
}

static GLXContext *
initGL(RubikGLWidget w) {
	XVisualInfo xviIn, *xviOut;
	int numVis;

	/*XGetWindowAttributes(XtDisplay(w), XtWindow(w), &xgwa); */
	xviIn.screen = DefaultScreen(XtDisplay(w));
	xviIn.visualid = XVisualIDFromVisual(DefaultVisual(XtDisplay(w),
		xviIn.screen));
	xviOut = XGetVisualInfo(XtDisplay(w), VisualScreenMask | VisualIDMask,
		&xviIn, &numVis);
	if (!xviOut) {
		XtWarning("Could not get XVisualInfo");
		return (GLXContext *) NULL;
	}
	if (glXContext)
		free(glXContext);
	if ((glXContext = (GLXContext *) malloc(sizeof (GLXContext))) ==
			NULL) {
		DISPLAY_ERROR("Not enough memory for glx info, exiting.");
	}
	*glXContext = glXCreateContext(XtDisplay(w), xviOut, 0, GL_TRUE);
	(void) XFree((char *) xviOut);
	if (!*glXContext) {
		XtWarning("Could not create rendering context");
		return (GLXContext *) NULL;
	}
	return glXContext;
}
#endif

#ifndef WINVER
static
#endif
void
resizePuzzleGL(RubikGLWidget w)
{
#ifdef WINVER
	RECT rect;

	/* Determine size of client area */
	(void) GetClientRect(w->core.hWnd, &rect);
	w->core.width = rect.right;
	w->core.height = rect.bottom;
#endif
	glViewport(0, 0, (GLint) w->core.width, (GLint) w->core.height);
	glMatrixMode(GL_PROJECTION);
	glLoadIdentity();
	glFrustum(-1.0, 1.0, -1.0, 1.0, 5.0, 15.0);
	glMatrixMode(GL_MODELVIEW);
}

static void
makeCurrentGL(RubikGLWidget w)
{
	GLboolean rgbaMode;

#ifdef WINVER
	wglMakeCurrent(w->core.hDC, hRC);
#else
	if (!glXMakeCurrent(XtDisplay(w), XtWindow(w), *glXContext)) {
		DISPLAY_WARNING("GL error");
	}
#endif
	madeCurrent = True;
	/* True Color junk */
	glGetBooleanv(GL_RGBA_MODE, &rgbaMode);
	if (!rgbaMode) {
#ifdef WINVER
		glClearIndex(0.0);
#else
		glIndexi(WhitePixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
		glClearIndex((float) BlackPixel(XtDisplay(w),
			DefaultScreen(XtDisplay(w))));
#endif
	}
	resizePuzzleGL(w);

	glDrawBuffer(GL_BACK);
	glClearDepth(1.0);
	glClearColor(face_material[NO_FACE][0], face_material[NO_FACE][1],
		face_material[NO_FACE][2], face_material[NO_FACE][3]);
	glColor3f(1.0, 1.0, 1.0);

	glLightfv(GL_LIGHT0, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT0, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT0, GL_POSITION, position0);
	glLightfv(GL_LIGHT1, GL_AMBIENT, ambient);
	glLightfv(GL_LIGHT1, GL_DIFFUSE, diffuse);
	glLightfv(GL_LIGHT1, GL_POSITION, position1);
	glLightModelfv(GL_LIGHT_MODEL_AMBIENT, lmodel_ambient);
	glLightModelfv(GL_LIGHT_MODEL_TWO_SIDE, lmodel_twoside);
	glEnable(GL_LIGHTING);
	glEnable(GL_LIGHT0);
	glEnable(GL_LIGHT1);
	glEnable(GL_DEPTH_TEST);
	glEnable(GL_NORMALIZE);
	glEnable(GL_CULL_FACE);
	glShadeModel(GL_FLAT);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SHININESS, front_shininess);
	glMaterialfv(GL_FRONT_AND_BACK, GL_SPECULAR, front_specular);

	w->rubikGL.step = 90;
	w->rubikGL.rotateStep = 0.0;
}

#ifndef WINVER
static void
initializePuzzleGL(Widget request, Widget renew)
{
	RubikGLWidget w = (RubikGLWidget) renew;
	int i;

	setAllColors((RubikWidget) w);
	(void) initGL(w);
	w->rubik.dim = 4; /* 2 and 3 already taken */
	for (i = 0; i < MAX_FACES + 2; i++)
		pickColor(w, i, w->rubik.mono, face_material[i]);
	w->rubikGL.numTurnInc = ((w->rubik.delay < MAX_TURN_INC) ?
		w->rubik.delay + 1 : MAX_TURN_INC);
	resizePuzzleGL(w);
}
#endif

#ifndef WINVER
static
#endif
void
exposePuzzleGL(
#ifdef WINVER
RubikGLWidget w
#else
Widget renew, XEvent *event, Region region
#endif
)
{
#ifdef WINVER
	if (hRC == NULL) {
		BYTE type;
		int i;

#if 1
		type = PFD_TYPE_RGBA;
#else
		type = PFD_TYPE_COLORINDEX;
#endif
		DWORD flags = PFD_DOUBLEBUFFER;
		(void) setupPixelFormat(w, type, flags);
		hRC = wglCreateContext(w->core.hDC);
		w->rubikGL.movement.face = NO_FACE;
		w->rubikGL.movement.position = NO_POSITION;
		w->rubikGL.movement.direction = NO_DEPTH;
		w->rubikGL.rotateStep = 0.0;
		w->rubikGL.angleStepX = 0.0;
		w->rubikGL.angleStepY = 0.0;
		w->rubikGL.angleStepZ = 0.0;
		w->rubik.dim = 4; /* 2 and 3 already taken */
		for (i = 0; i < MAX_FACES + 2; i++)
			pickColor(w, i, w->rubik.mono, face_material[i]);
		}
		w->rubikGL.numTurnInc = ((w->rubik.delay < MAX_TURN_INC) ?
			w->rubik.delay + 1 : MAX_TURN_INC);
		resizePuzzleGL(w);
#else
	RubikGLWidget w = (RubikGLWidget) renew;

	if (!w->core.visible)
		return;
#endif
	if (!madeCurrent) {
		makeCurrentGL(w);
	}
	drawFrameGL(w, w->rubik.focus);
	drawAllPiecesGL(w);
}

#ifndef WINVER
static void
movePuzzleTl(RubikGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		TL, (int) (event->xkey.state & ControlMask));
}

static void
movePuzzleTop(RubikGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		TOP, (int) (event->xkey.state & ControlMask));
}

static void
movePuzzleTr(RubikGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		TR, (int) (event->xkey.state & ControlMask));
}

static void
movePuzzleLeft(RubikGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		LEFT, (int) (event->xkey.state & ControlMask));
}

static void
movePuzzleRight(RubikGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		RIGHT, (int) (event->xkey.state & ControlMask));
}

static void
movePuzzleBl(RubikGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		BL, (int) (event->xkey.state & ControlMask));
}

static void
movePuzzleBottom(RubikGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		BOTTOM, (int) (event->xkey.state & ControlMask));
}

static void
movePuzzleBr(RubikGLWidget w, XEvent *event, char **args, int nArgs)
{
	movePuzzleInput((RubikWidget) w, event->xbutton.x, event->xbutton.y,
		BR, (int) (event->xkey.state & ControlMask));
}
#endif
#endif
